{ This unit defines the structure of the model. There are four functions. The
  first function, called counts, defines the number, names, and units of the
  model, the state variables, the process variables, the driver variables and
  the parameters. The second function, called processes, is the actual equations
  which make up the model. The third function, derivs, calculates the
  derivatives of state variables. And the fourth function, parcount, is used to
  automatically number the parameters consecutively. 
    The state variables, driver variables, process variables and parameters are
  all stored in global arrays, called stat, drive, proc, and par, respectively.
  The function counts accesses the global arrays directly but the other functions
  operate on copies of the global arrays. }
unit equations;

interface

uses  stypes, math;

PROCEDURE counts;
PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
function ParCount(processnum:integer) : integer;

var
  tproc:processarray;
  tstat:statearray;
  sensflag:boolean;
  newyear:Boolean = false;
  DayofYear: double = 0;

implementation

uses frontend, calculate, options;

           { Do not make modifcations above this line. }
{*****************************************************************************}

{ This procedure defines the model. The number of parameters, state, driver and
  process variables are all set in this procedure. The model name, version
  number and time unit are also set here. This procedure accesses the global
  arrays containing the the parameters, state, driver and process variables and
  the global structure ModelDef directly, to save memory space. }
PROCEDURE counts;
var
 i,npar,CurrentProc:integer;
begin
{ Set the modelname, version and time unit. }
ModelDef.modelname := 'MELIV';
ModelDef.versionnumber := '1.0.0';
ModelDef.timeunit := 'day';
ModelDef.contactperson := 'Ed Rastetter';
ModelDef.contactaddress1 := 'The Ecosystems Center';
ModelDef.contactaddress2 := 'Marine Biological Laboratory';
ModelDef.contactaddress3 := 'Woods Hole, MA 02543';

{ Set the number of state variables in the model. The maximum number of state
  variables is maxstate, in unit stypes. }
ModelDef.numstate := 67;

{ Enter the name, units and symbol for each state variable. The maximum length
  of the state variable name is 17 characters and the maximum length for units
  and symbol is stringlength (specified in unit stypes) characters. }
 
 
with stat[1] do
 begin
    name:='biomass C';  units:='g C m-2';  symbol:='BC';
 end;
 
with stat[2] do
 begin
    name:='biomass N';  units:='g N m-2';  symbol:='BN';
 end;
 
with stat[3] do
 begin
    name:='biomass P';  units:='g P m-2';  symbol:='BP';
 end;
 
with stat[4] do
 begin
    name:='effort C';  units:='effort g-1 DW';  symbol:='VC';
 end;
 
with stat[5] do
 begin
    name:='effort NH4';  units:='effort g-1 DW';  symbol:='VNH4';
 end;
 
with stat[6] do
 begin
    name:='effort NO3';  units:='effort g-1 DW';  symbol:='VNO3';
 end;
 
with stat[7] do
 begin
    name:='effort PO4';  units:='effort g-1 DW';  symbol:='VP';
 end;
 
with stat[8] do
 begin
    name:='effort light';  units:='effort g-1 DW';  symbol:='VI';
 end;
 
with stat[9] do
 begin
    name:='effort H2O';  units:='effort g-1 DW';  symbol:='VW';
 end;
 
with stat[10] do
 begin
    name:='effort DON';  units:='effort g-2 DW';  symbol:='VDON';
 end;
 
with stat[11] do
 begin
    name:='effort Nfix';  units:='effort g-2 DW';  symbol:='VNfix';
 end;
 
with stat[12] do
 begin
    name:='soil organic C';  units:='gC m-2';  symbol:='DC';
 end;
 
with stat[13] do
 begin
    name:='soil organic N';  units:='gC m-2';  symbol:='DN';
 end;
 
with stat[14] do
 begin
    name:='soil organic P';  units:='gC m-2';  symbol:='DP';
 end;
 
with stat[15] do
 begin
    name:='Woody debris C';  units:='gC m-2';  symbol:='WC';
 end;
 
with stat[16] do
 begin
    name:='Woody debris N';  units:='gC m-2';  symbol:='WN';
 end;
 
with stat[17] do
 begin
    name:='Woody debris P';  units:='gC m-2';  symbol:='WP';
 end;
 
with stat[18] do
 begin
    name:='Phase II SOM C';  units:='gC m-2';  symbol:='SC';
 end;
 
with stat[19] do
 begin
    name:='Phase II SOM N';  units:='gC m-2';  symbol:='SN';
 end;
 
with stat[20] do
 begin
    name:='Phase II SOM P';  units:='gC m-2';  symbol:='SP';
 end;
 
with stat[21] do
 begin
    name:='NH4';  units:='g N m-2';  symbol:='ENH4';
 end;
 
with stat[22] do
 begin
    name:='NO3';  units:='g N m-2';  symbol:='ENO3';
 end;
 
with stat[23] do
 begin
    name:='P PO4';  units:='g P m-2';  symbol:='EPO4';
 end;
 
with stat[24] do
 begin
    name:='P Prim min';  units:='g P m-2';  symbol:='PA';
 end;
 
with stat[25] do
 begin
    name:='P non-Occluded';  units:='g P m-2';  symbol:='Pno';
 end;
 
with stat[26] do
 begin
    name:='P Occluded';  units:='g P m-2';  symbol:='Poccl';
 end;
 
with stat[27] do
 begin
    name:='g labile DOM C';  units:='g C m-2';  symbol:='EDOM';
 end;
 
with stat[28] do
 begin
    name:='g recalcitrant DOM C';  units:='g C m-2';  symbol:='EDOMR';
 end;
 
with stat[29] do
 begin
    name:='water';  units:='mm';  symbol:='W';
 end;
 
with stat[30] do
 begin
    name:='snow pack';  units:='mm H2O';  symbol:='Snow';
 end;
 
with stat[31] do
 begin
    name:='cum interception';  units:='mm';  symbol:='cumint';
 end;
 
with stat[32] do
 begin
    name:='Ave requirement C';  units:='none';  symbol:='RA_C';
 end;
 
with stat[33] do
 begin
    name:='Ave requirement N';  units:='none';  symbol:='RA_N';
 end;
 
with stat[34] do
 begin
    name:='Ave requirement NH4';  units:='none';  symbol:='RA_NH4';
 end;
 
with stat[35] do
 begin
    name:='Ave requirement NO3';  units:='none';  symbol:='RA_NO3';
 end;
 
with stat[36] do
 begin
    name:='Ave requirement DON';  units:='none';  symbol:='RA_DON';
 end;
 
with stat[37] do
 begin
    name:='Ave requirement Nfix';  units:='none';  symbol:='RA_Nfix';
 end;
 
with stat[38] do
 begin
    name:='Ave requirement P';  units:='none';  symbol:='RA_P';
 end;
 
with stat[39] do
 begin
    name:='Ave requirement H2O';  units:='none';  symbol:='RA_W';
 end;
 
with stat[40] do
 begin
    name:='Ave Acquisition C';  units:='none';  symbol:='UA_C';
 end;
 
with stat[41] do
 begin
    name:='Ave Acquisition NH4';  units:='none';  symbol:='UA_NH4';
 end;
 
with stat[42] do
 begin
    name:='Ave Acquisition NO3';  units:='none';  symbol:='UA_NO3';
 end;
 
with stat[43] do
 begin
    name:='Ave Acquisition PO4';  units:='none';  symbol:='UA_P';
 end;
 
with stat[44] do
 begin
    name:='Ave Acquisition light';  units:='none';  symbol:='UA_I';
 end;
 
with stat[45] do
 begin
    name:='Ave Acquisition H2O';  units:='none';  symbol:='UA_W';
 end;
 
with stat[46] do
 begin
    name:='Ave Acquisition DON';  units:='none';  symbol:='UA_DON';
 end;
 
with stat[47] do
 begin
    name:='Ave Acquisition Nfix';  units:='none';  symbol:='UA_Nfix';
 end;
 
with stat[48] do
 begin
    name:='cum GPP';  units:='gC m-2';  symbol:='cumGPP';
 end;
 
with stat[49] do
 begin
    name:='cum NPP';  units:='gC m-2';  symbol:='cumNPP';
 end;
 
with stat[50] do
 begin
    name:='cum litter C';  units:='gC m-2';  symbol:='cumlitC';
 end;
 
with stat[51] do
 begin
    name:='cum microb resp';  units:='gC m-2';  symbol:='cumRm';
 end;
 
with stat[52] do
 begin
    name:='cum N uptake';  units:='gN m-2';  symbol:='cumNup';
 end;
 
with stat[53] do
 begin
    name:='cum litter N';  units:='gN m-2';  symbol:='cumlitN';
 end;
 
with stat[54] do
 begin
    name:='cum net N min';  units:='gN m-2';  symbol:='cumNmin';
 end;
 
with stat[55] do
 begin
    name:='cum P uptake';  units:='gP m-2';  symbol:='cumPup';
 end;
 
with stat[56] do
 begin
    name:='cum litter P';  units:='gP m-2';  symbol:='cumlitP';
 end;
 
with stat[57] do
 begin
    name:='cum net P min';  units:='gP m-2';  symbol:='cumPmin';
 end;
 
with stat[58] do
 begin
    name:='cum precipitation';  units:='mm H2O';  symbol:='cumPpt';
 end;
 
with stat[59] do
 begin
    name:='cum ET';  units:='mm H2O';  symbol:='cumET';
 end;
 
with stat[60] do
 begin
    name:='cum runoff';  units:='mm H2O';  symbol:='cumRO';
 end;
 
with stat[61] do
 begin
    name:='cum N inputs';  units:='g N m-2';  symbol:='cumNin';
 end;
 
with stat[62] do
 begin
    name:='cum N losses';  units:='g N m-2';  symbol:='cumNout';
 end;
 
with stat[63] do
 begin
    name:='cum P inputs';  units:='g P m-2';  symbol:='cumPin';
 end;
 
with stat[64] do
 begin
    name:='cum P losses';  units:='g P m-2';  symbol:='cumPout';
 end;
 
with stat[65] do
 begin
    name:='cum N fixation';  units:='g P m-2';  symbol:='cumNfix';
 end;
 
with stat[66] do
 begin
    name:='degree day';  units:='oC day';  symbol:='DDay';
 end;
 
with stat[67] do
 begin
    name:='Cum N_S Nfix';  units:='gN m-2';  symbol:='cumNSNfix';
 end;

{ Set the total number of processes in the model. The first numstate processes
  are the derivatives of the state variables. The maximum number of processes is
  maxparam, in unit stypes. }
ModelDef.numprocess := ModelDef.numstate + 163;

{ For each process, set proc[i].parameters equal to the number of parameters
  associated with that process, and set IsDiscrete to true or false. After each
  process, set the name, units, and symbol for all parameters associated with
  that process. Note that Parcount returns the total number of parameters in
  all previous processes. }
 
CurrentProc := ModelDef.numstate + 1;
With proc[CurrentProc] do
   begin
      name       := 'Temperature ave';
      units       := 'oC';
      symbol       := 'Ta';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 2;
With proc[CurrentProc] do
   begin
      name       := 'Biomass';
      units       := 'g DW m-2';
      symbol       := 'BT';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 3;
With proc[CurrentProc] do
   begin
      name       := 'Biomass Active';
      units       := 'g DW m-2';
      symbol       := 'BA';
      parameters       := 4;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Max BA';  units:='gDW m-2';  symbol:='BAmax';
 end;
with par[npar + 2] do
 begin
    name:='BA:BT at low BT';  units:='none';  symbol:='gammaB';
 end;
with par[npar + 3] do
 begin
    name:='leaf area per DW';  units:='m2 g-1 DW';  symbol:='a_sla';
 end;
with par[npar + 4] do
 begin
    name:='C:DW ratio';  units:='g C g-1 DW';  symbol:='qC';
 end;
 
CurrentProc := ModelDef.numstate + 4;
With proc[CurrentProc] do
   begin
      name       := 'Biomass leaf';
      units       := 'g DW m-2';
      symbol       := 'BL';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 5;
With proc[CurrentProc] do
   begin
      name       := 'Biomass root';
      units       := 'g DW m-2';
      symbol       := 'BR';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 6;
With proc[CurrentProc] do
   begin
      name       := 'Biomass woody';
      units       := 'g DW m-2';
      symbol       := 'BW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 7;
With proc[CurrentProc] do
   begin
      name       := 'leaf area';
      units       := 'm2 m-2';
      symbol       := 'L';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 8;
With proc[CurrentProc] do
   begin
      name       := 'Leaf area max';
      units       := 'm2 m-2';
      symbol       := 'L_max';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 9;
With proc[CurrentProc] do
   begin
      name       := 'canopy fraction';
      units       := 'm2 m-2';
      symbol       := 'fc';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 10;
With proc[CurrentProc] do
   begin
      name       := 'VPD';
      units       := 'MPa';
      symbol       := 'DELTA_E';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 11;
With proc[CurrentProc] do
   begin
      name       := 'Avail. water';
      units       := 'none';
      symbol       := 'AW';
      parameters       := 24;
      ptype       := ptGroup5;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='field capacity';  units:='fraction of soil vol';  symbol:='theta_f';
 end;
with par[npar + 2] do
 begin
    name:='wilting point';  units:='fraction of soil vol';  symbol:='theta_w';
 end;
with par[npar + 3] do
 begin
    name:='field potential';  units:='MPa';  symbol:='psi_f';
 end;
with par[npar + 4] do
 begin
    name:='wilting potential';  units:='MPa';  symbol:='psi_w';
 end;
with par[npar + 5] do
 begin
    name:='soil depth';  units:='m';  symbol:='z';
 end;
with par[npar + 6] do
 begin
    name:='soil porosity';  units:='fraction';  symbol:='pore';
 end;
with par[npar + 7] do
 begin
    name:='bulk density';  units:='Mg m-3';  symbol:='rho_S';
 end;
with par[npar + 8] do
 begin
    name:='soil drain rate';  units:='day-1';  symbol:='drain';
 end;
with par[npar + 9] do
 begin
    name:='non-leaf sfc area';  units:='m2 m-2';  symbol:='NLsfc';
 end;
with par[npar + 10] do
 begin
    name:='branch exponent';  units:='none';  symbol:='NLe';
 end;
with par[npar + 11] do
 begin
    name:='Mid wood biomass';  units:='g DW m-2';  symbol:='MBW';
 end;
with par[npar + 12] do
 begin
    name:='intercept volume';  units:='mm m-2';  symbol:='IntV';
 end;
with par[npar + 13] do
 begin
    name:='NH4 sorption cap';  units:='g NH4-N Mg-1 dry soil';  symbol:='SNH4';
 end;
with par[npar + 14] do
 begin
    name:='NH4 affinity const';  units:='umol NH4 L-1';  symbol:='etaNH4';
 end;
with par[npar + 15] do
 begin
    name:='DOM sorption cap';  units:='g DOM-M Mg-1 dry soil';  symbol:='SDOM';
 end;
with par[npar + 16] do
 begin
    name:='DOM affinity const';  units:='umol DOM L-1';  symbol:='etaDOM';
 end;
with par[npar + 17] do
 begin
    name:='DOMR sorption cap';  units:='g DOM-M Mg-1 dry soil';  symbol:='SDOMR';
 end;
with par[npar + 18] do
 begin
    name:='DOMR affinity const';  units:='umol DOM L-1';  symbol:='etaDOMR';
 end;
with par[npar + 19] do
 begin
    name:='PO4 partition const';  units:='g sorbed g-1 solution';  symbol:='etaPO4';
 end;
with par[npar + 20] do
 begin
    name:='runoff frac NH4';  units:='none';  symbol:='ROfNH4';
 end;
with par[npar + 21] do
 begin
    name:='runoff frac NO3';  units:='none';  symbol:='ROfNO3';
 end;
with par[npar + 22] do
 begin
    name:='runoff frac DOM';  units:='none';  symbol:='ROfDOM';
 end;
with par[npar + 23] do
 begin
    name:='runoff frac DOMR';  units:='none';  symbol:='ROfDOMR';
 end;
with par[npar + 24] do
 begin
    name:='runoff frac PO4';  units:='none';  symbol:='ROfPO4';
 end;
 
CurrentProc := ModelDef.numstate + 12;
With proc[CurrentProc] do
   begin
      name       := 'P bal Apatite weatherin';
      units       := 'g P m-2 day-1';
      symbol       := 'PaW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 13;
With proc[CurrentProc] do
   begin
      name       := 'P bal PO4 precipitation';
      units       := 'g P m-2 day-1';
      symbol       := 'PO4P';
      parameters       := 6;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='apatite weathering rate';  units:='day-1';  symbol:='rPaW';
 end;
with par[npar + 2] do
 begin
    name:='Max non-occlud';  units:='g P Mg-1 dry soil';  symbol:='Pnomax';
 end;
with par[npar + 3] do
 begin
    name:='PO4 precip rate';  units:='day-1';  symbol:='rPO4P';
 end;
with par[npar + 4] do
 begin
    name:='non-occld sol rate';  units:='day-1';  symbol:='rPnoW';
 end;
with par[npar + 5] do
 begin
    name:='No-occld stab rate';  units:='day-1';  symbol:='rPnoS';
 end;
with par[npar + 6] do
 begin
    name:='occlud weathering rate';  units:='day-1';  symbol:='rPocclW';
 end;
 
CurrentProc := ModelDef.numstate + 14;
With proc[CurrentProc] do
   begin
      name       := 'P bal Non-occl weathrng';
      units       := 'g P m-2 day-1';
      symbol       := 'PnoW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 15;
With proc[CurrentProc] do
   begin
      name       := 'P bal occld weathering';
      units       := 'g P m-2 day-1';
      symbol       := 'PocclW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 16;
With proc[CurrentProc] do
   begin
      name       := 'P bal non-occld stablz';
      units       := 'g P m-2 day-1';
      symbol       := 'PnoS';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 17;
With proc[CurrentProc] do
   begin
      name       := 'soil potential';
      units       := 'MPa';
      symbol       := 'psiS';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 18;
With proc[CurrentProc] do
   begin
      name       := 'canopy cond max';
      units       := 'm day-1';
      symbol       := 'c_cmax';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 19;
With proc[CurrentProc] do
   begin
      name       := 'canopy cond soil limite';
      units       := 'm day-1';
      symbol       := 'c_cs';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 20;
With proc[CurrentProc] do
   begin
      name       := 'canopy cond light opt';
      units       := 'm day-1';
      symbol       := 'c_cI';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 21;
With proc[CurrentProc] do
   begin
      name       := 'canopy cond';
      units       := 'm day-1';
      symbol       := 'c_c';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 22;
With proc[CurrentProc] do
   begin
      name       := 'photosynthesis CO2';
      units       := 'g C m-2 day-1';
      symbol       := 'UCc';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 23;
With proc[CurrentProc] do
   begin
      name       := 'photosynthesis light';
      units       := 'g C m-2 day-1';
      symbol       := 'UCI';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 24;
With proc[CurrentProc] do
   begin
      name       := 'photosynthesis water';
      units       := 'g C m-2 day-1';
      symbol       := 'UCW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 25;
With proc[CurrentProc] do
   begin
      name       := 'C bal photosynthesis';
      units       := 'g C m-2 day-1';
      symbol       := 'UC';
      parameters       := 11;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Ps CO2 rate constant';  units:='g C effort-1 day-1';  symbol:='gC';
 end;
with par[npar + 2] do
 begin
    name:='1/2 sat CA';  units:='ppm';  symbol:='kC';
 end;
with par[npar + 3] do
 begin
    name:='Tmax Ps CA';  units:='oC';  symbol:='TxC';
 end;
with par[npar + 4] do
 begin
    name:='Topt Ps CA';  units:='oC';  symbol:='ToC';
 end;
with par[npar + 5] do
 begin
    name:='T exp Ps CA';  units:='oC-1';  symbol:='aC';
 end;
with par[npar + 6] do
 begin
    name:='Ps light constant';  units:='g C effort-1 day-1';  symbol:='gI';
 end;
with par[npar + 7] do
 begin
    name:='quantum yeild';  units:='gC MJ-1';  symbol:='E0';
 end;
with par[npar + 8] do
 begin
    name:='light extinct';  units:='none';  symbol:='kI';
 end;
with par[npar + 9] do
 begin
    name:='Tmax Ps light';  units:='oC';  symbol:='TxL';
 end;
with par[npar + 10] do
 begin
    name:='Topt Ps light';  units:='oC';  symbol:='ToL';
 end;
with par[npar + 11] do
 begin
    name:='T exp Ps light';  units:='oC-1';  symbol:='aL';
 end;
 
CurrentProc := ModelDef.numstate + 26;
With proc[CurrentProc] do
   begin
      name       := 'W bal water uptake';
      units       := 'mm day-1';
      symbol       := 'UW';
      parameters       := 4;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='H2O uptake constant';  units:='mm/MPa/m/day';  symbol:='gW';
 end;
with par[npar + 2] do
 begin
    name:='PET 1/2 sat par';  units:='mm day';  symbol:='kE';
 end;
with par[npar + 3] do
 begin
    name:='max leaf cond';  units:='m day-1';  symbol:='c_smax';
 end;
with par[npar + 4] do
 begin
    name:='Ca H2O weighting';  units:='none';  symbol:='beta';
 end;
 
CurrentProc := ModelDef.numstate + 27;
With proc[CurrentProc] do
   begin
      name       := 'potential ET';
      units       := 'mm day-1';
      symbol       := 'UWp';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 28;
With proc[CurrentProc] do
   begin
      name       := 'aqueous NH4';
      units       := 'umol NH4 L-1';
      symbol       := 'NH4aq';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 29;
With proc[CurrentProc] do
   begin
      name       := 'N bal NH4 uptake';
      units       := 'gN m-2 day-1';
      symbol       := 'UNH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 30;
With proc[CurrentProc] do
   begin
      name       := 'aqueous NO3';
      units       := 'umol NO3 L-1';
      symbol       := 'NO3aq';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 31;
With proc[CurrentProc] do
   begin
      name       := 'N bal NO3 uptake';
      units       := 'gN m-2 day-1';
      symbol       := 'UNO3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 32;
With proc[CurrentProc] do
   begin
      name       := 'Aq labile DOM';
      units       := 'umol C L-1';
      symbol       := 'DOM';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 33;
With proc[CurrentProc] do
   begin
      name       := 'Aq recalcitrant DOM';
      units       := 'umol C L-1';
      symbol       := 'DOMR';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 34;
With proc[CurrentProc] do
   begin
      name       := 'DOM uptake';
      units       := 'gC m-2 day-1';
      symbol       := 'UDOM';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 35;
With proc[CurrentProc] do
   begin
      name       := 'N bal DON uptake';
      units       := 'gN m-2 day-1';
      symbol       := 'UDON';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 36;
With proc[CurrentProc] do
   begin
      name       := 'N bal N fixation';
      units       := 'gN m-2 day-1';
      symbol       := 'UNfix';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 37;
With proc[CurrentProc] do
   begin
      name       := 'N bal total N uptake';
      units       := 'gN m-2 day-1';
      symbol       := 'UNtot';
      parameters       := 15;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='NH4 uptake const';  units:='gN effort-1 day-1';  symbol:='gNH4';
 end;
with par[npar + 2] do
 begin
    name:='NH4 1/2 sat const';  units:='umol L-1';  symbol:='kNH4';
 end;
with par[npar + 3] do
 begin
    name:='Q10 NH4 uptake';  units:='none';  symbol:='Q10NH4';
 end;
with par[npar + 4] do
 begin
    name:='NO3 uptake const';  units:='gN effort-1 day-1';  symbol:='gNO3';
 end;
with par[npar + 5] do
 begin
    name:='NO31/2 sat const';  units:='umol L-1';  symbol:='kNO3';
 end;
with par[npar + 6] do
 begin
    name:='Q10 NO3 uptake';  units:='none';  symbol:='Q10NO3';
 end;
with par[npar + 7] do
 begin
    name:='DOM uptake const';  units:='gC effort-1 day-1';  symbol:='gDOM';
 end;
with par[npar + 8] do
 begin
    name:='DOM 1/2 sat const';  units:='umol L-1';  symbol:='kDOM';
 end;
with par[npar + 9] do
 begin
    name:='Q10 DOM uptake';  units:='none';  symbol:='Q10DOM';
 end;
with par[npar + 10] do
 begin
    name:='N fix rate const';  units:='gN effort-1 day-1';  symbol:='gNfix';
 end;
with par[npar + 11] do
 begin
    name:='Q10 N fix';  units:='none';  symbol:='Q10Nfix';
 end;
with par[npar + 12] do
 begin
    name:='NH4 C cost';  units:='gC g-1 N';  symbol:='NH4Ccost';
 end;
with par[npar + 13] do
 begin
    name:='NO3 C cost';  units:='gC g-1 N';  symbol:='NO3Ccost';
 end;
with par[npar + 14] do
 begin
    name:='DON C cost';  units:='gC g-1 N';  symbol:='DONCcost';
 end;
with par[npar + 15] do
 begin
    name:='N fix C cost';  units:='gC g-1 N';  symbol:='NfixCcost';
 end;
 
CurrentProc := ModelDef.numstate + 38;
With proc[CurrentProc] do
   begin
      name       := 'aqueous PO4';
      units       := 'umol PO4 L-1';
      symbol       := 'PO4aq';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 39;
With proc[CurrentProc] do
   begin
      name       := 'P bal PO4 uptake';
      units       := 'gP m-2 day-1';
      symbol       := 'UPO4';
      parameters       := 3;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='PO4 uptake const';  units:='gP effort-1 day-1';  symbol:='gPO4';
 end;
with par[npar + 2] do
 begin
    name:='PO4 1/2 sat const';  units:='umol L-1';  symbol:='kPO4';
 end;
with par[npar + 3] do
 begin
    name:='Q10 PO4 uptake';  units:='none';  symbol:='Q10PO4';
 end;
 
CurrentProc := ModelDef.numstate + 40;
With proc[CurrentProc] do
   begin
      name       := 'actual N conc.';
      units       := 'g N g-1 DW';
      symbol       := 'aqN';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 41;
With proc[CurrentProc] do
   begin
      name       := 'optimum N conc.';
      units       := 'g N g-1 DW';
      symbol       := 'qN';
      parameters       := 7;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='leaf N:DW';  units:='g N g-1 DW';  symbol:='qLN';
 end;
with par[npar + 2] do
 begin
    name:='Wood N:DW';  units:='g N g-1 DW';  symbol:='qWN';
 end;
with par[npar + 3] do
 begin
    name:='root N:DW';  units:='g N g-1 DW';  symbol:='qRN';
 end;
with par[npar + 4] do
 begin
    name:='leaf litter N:DW';  units:='g N g-1 DW';  symbol:='qLNl';
 end;
with par[npar + 5] do
 begin
    name:='wood litter N:DW';  units:='g N g-1 DW';  symbol:='qWNl';
 end;
with par[npar + 6] do
 begin
    name:='root litter N:DW';  units:='g N g-1 DW';  symbol:='qRNl';
 end;
with par[npar + 7] do
 begin
    name:='stoichio feedback';  units:='none';  symbol:='kq';
 end;
 
CurrentProc := ModelDef.numstate + 42;
With proc[CurrentProc] do
   begin
      name       := 'actual P conc.';
      units       := 'g P g-1 DW';
      symbol       := 'aqP';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 43;
With proc[CurrentProc] do
   begin
      name       := 'optimum P conc.';
      units       := 'g P g-1 DW';
      symbol       := 'qP';
      parameters       := 6;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='leaf P:DW';  units:='g P g-1 DW';  symbol:='qLP';
 end;
with par[npar + 2] do
 begin
    name:='Wood P:DW';  units:='g P g-1 DW';  symbol:='qWP';
 end;
with par[npar + 3] do
 begin
    name:='root P:DW';  units:='g P g-1 DW';  symbol:='qRP';
 end;
with par[npar + 4] do
 begin
    name:='leaf litter P:DW';  units:='g P g-1 DW';  symbol:='qLPl';
 end;
with par[npar + 5] do
 begin
    name:='wood litter P:DW';  units:='g P g-1 DW';  symbol:='qWPl';
 end;
with par[npar + 6] do
 begin
    name:='root litter P:DW';  units:='g P g-1 DW';  symbol:='qRPl';
 end;
 
CurrentProc := ModelDef.numstate + 44;
With proc[CurrentProc] do
   begin
      name       := 'C bal Litter C';
      units       := 'g C m-2 day-1';
      symbol       := 'LitC';
      parameters       := 4;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='leaf turnover evergreen';  units:='day-1';  symbol:='mAL';
 end;
with par[npar + 2] do
 begin
    name:='BW turnover';  units:='day-1';  symbol:='mW';
 end;
with par[npar + 3] do
 begin
    name:='root turnover';  units:='day-1';  symbol:='mAR';
 end;
with par[npar + 4] do
 begin
    name:='min canopy fraction';  units:='none';  symbol:='fracmin';
 end;
 
CurrentProc := ModelDef.numstate + 45;
With proc[CurrentProc] do
   begin
      name       := 'N bal Litter N';
      units       := 'g N m-2 day-1';
      symbol       := 'LitN';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 46;
With proc[CurrentProc] do
   begin
      name       := 'P bal Litter P';
      units       := 'g P m-2 day-1';
      symbol       := 'LitP';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 47;
With proc[CurrentProc] do
   begin
      name       := 'C coarse litter';
      units       := 'g C m-2 day-1';
      symbol       := 'LCWC';
      parameters       := 4;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='coarse woody turnover';  units:='day-1';  symbol:='mCW';
 end;
with par[npar + 2] do
 begin
    name:='wood litter exp';  units:='none';  symbol:='mCWex';
 end;
with par[npar + 3] do
 begin
    name:='coarse litter N:DW';  units:='gN g-1 DW';  symbol:='qWNwl';
 end;
with par[npar + 4] do
 begin
    name:='coarse litter P:DW';  units:='gP g-1 DW';  symbol:='qWPwl';
 end;
 
CurrentProc := ModelDef.numstate + 48;
With proc[CurrentProc] do
   begin
      name       := 'N coarse litter';
      units       := 'g N m-2 day-1';
      symbol       := 'LCWN';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 49;
With proc[CurrentProc] do
   begin
      name       := 'P coarse litter';
      units       := 'g P m-2 day-1';
      symbol       := 'LCWP';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 50;
With proc[CurrentProc] do
   begin
      name       := 'C CWD activation';
      units       := 'g C m-2 day-1';
      symbol       := 'LCWCa';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='CWD turnover';  units:='day-1';  symbol:='omega';
 end;
 
CurrentProc := ModelDef.numstate + 51;
With proc[CurrentProc] do
   begin
      name       := 'N CWD activation';
      units       := 'g N m-2 day-1';
      symbol       := 'LCWNa';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 52;
With proc[CurrentProc] do
   begin
      name       := 'P CWD activation';
      units       := 'g P m-2 day-1';
      symbol       := 'LCWPa';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 53;
With proc[CurrentProc] do
   begin
      name       := 'maint C req';
      units       := 'gC m-2 day-1';
      symbol       := 'RCm';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 54;
With proc[CurrentProc] do
   begin
      name       := 'C bal total plant resp';
      units       := 'g C m-2 day-1';
      symbol       := 'RCPt';
      parameters       := 5;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='active resp';  units:='gC g-1 N day-1';  symbol:='rmA';
 end;
with par[npar + 2] do
 begin
    name:='woody resp';  units:='gC g-1 N day-1';  symbol:='rmW';
 end;
with par[npar + 3] do
 begin
    name:='sapwd:heartwd parti exp';  units:='m2 g-1 DW';  symbol:='krmW';
 end;
with par[npar + 4] do
 begin
    name:='Q10 respiration';  units:='none';  symbol:='Q10R';
 end;
with par[npar + 5] do
 begin
    name:='growth resp';  units:='fraction';  symbol:='rgrow';
 end;
 
CurrentProc := ModelDef.numstate + 55;
With proc[CurrentProc] do
   begin
      name       := 'N-use eff';
      units       := 'gDW g-1 N';
      symbol       := 'NUE';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 56;
With proc[CurrentProc] do
   begin
      name       := 'P-use eff';
      units       := 'gDW g-1 P';
      symbol       := 'PUE';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 57;
With proc[CurrentProc] do
   begin
      name       := 'H2O-use eff';
      units       := 'gDW g-1 P';
      symbol       := 'WUE';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 58;
With proc[CurrentProc] do
   begin
      name       := 'allocatable effort';
      units       := 'effort g-1 DW';
      symbol       := 'Vstar';
      parameters       := 5;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='acclim rate';  units:='day-1';  symbol:='a';
 end;
with par[npar + 2] do
 begin
    name:='requirement turnover';  units:='day-1';  symbol:='tau';
 end;
with par[npar + 3] do
 begin
    name:='effort surcharge';  units:='effort';  symbol:='Vsc';
 end;
with par[npar + 4] do
 begin
    name:='N-yield variance';  units:='none';  symbol:='epsilon';
 end;
with par[npar + 5] do
 begin
    name:='max req to up index';  units:='none';  symbol:='lambda';
 end;
 
CurrentProc := ModelDef.numstate + 59;
With proc[CurrentProc] do
   begin
      name       := 'growth C req';
      units       := 'gC m-2 day-1';
      symbol       := 'RCg';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 60;
With proc[CurrentProc] do
   begin
      name       := 'growth N req';
      units       := 'gN m-2 day-1';
      symbol       := 'RNg';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 61;
With proc[CurrentProc] do
   begin
      name       := 'growth P req';
      units       := 'gP m-2 day-1';
      symbol       := 'RPg';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 62;
With proc[CurrentProc] do
   begin
      name       := 'yield NH4';
      units       := 'g N m-2 day-1';
      symbol       := 'yNH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 63;
With proc[CurrentProc] do
   begin
      name       := 'yield NO3';
      units       := 'g N m-2 day-1';
      symbol       := 'yNO3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 64;
With proc[CurrentProc] do
   begin
      name       := 'yield DON';
      units       := 'g N m-2 day-1';
      symbol       := 'yDON';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 65;
With proc[CurrentProc] do
   begin
      name       := 'yield Nfixt';
      units       := 'g N m-2 day-1';
      symbol       := 'yNfix';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 66;
With proc[CurrentProc] do
   begin
      name       := 'yield x uptake';
      units       := '(gNm-2day-1)2';
      symbol       := 'sumUy';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 67;
With proc[CurrentProc] do
   begin
      name       := 'Total C req';
      units       := 'g C m-2 day-1';
      symbol       := 'RC';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 68;
With proc[CurrentProc] do
   begin
      name       := 'Total PO4 req';
      units       := 'g P m-2 day-1';
      symbol       := 'RP';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 69;
With proc[CurrentProc] do
   begin
      name       := 'Total N req';
      units       := 'g N m-2 day-1';
      symbol       := 'RNT';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 70;
With proc[CurrentProc] do
   begin
      name       := 'Total NH4 req';
      units       := 'g N m-2 day-1';
      symbol       := 'RNH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 71;
With proc[CurrentProc] do
   begin
      name       := 'Total NO3 req';
      units       := 'g N m-2 day-1';
      symbol       := 'RNO3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 72;
With proc[CurrentProc] do
   begin
      name       := 'Total DON req';
      units       := 'g N m-2 day-1';
      symbol       := 'RDON';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 73;
With proc[CurrentProc] do
   begin
      name       := 'Total Nfix req';
      units       := 'g N m-2 day-1';
      symbol       := 'RNfix';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 74;
With proc[CurrentProc] do
   begin
      name       := 'Mean req:uptake';
      units       := 'none';
      symbol       := 'PHI';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 75;
With proc[CurrentProc] do
   begin
      name       := 'mesophyll CA';
      units       := 'ppm';
      symbol       := 'Ci';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 76;
With proc[CurrentProc] do
   begin
      name       := 'Total effort';
      units       := 'none';
      symbol       := 'VTot';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 77;
With proc[CurrentProc] do
   begin
      name       := 'Total root effort';
      units       := 'none';
      symbol       := 'VR';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 78;
With proc[CurrentProc] do
   begin
      name       := 'Total leaf effort';
      units       := 'none';
      symbol       := 'VL';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 79;
With proc[CurrentProc] do
   begin
      name       := 'C bal NPP';
      units       := 'g C m-2 day-1';
      symbol       := 'NPP';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 80;
With proc[CurrentProc] do
   begin
      name       := 'W bal runoff';
      units       := 'mm day-1';
      symbol       := 'RO';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 81;
With proc[CurrentProc] do
   begin
      name       := 'NH4 leach';
      units       := 'gN m-2 day-1';
      symbol       := 'LNH4';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 82;
With proc[CurrentProc] do
   begin
      name       := 'NO3 leach';
      units       := 'gN m-2 day-1';
      symbol       := 'LNO3';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 83;
With proc[CurrentProc] do
   begin
      name       := 'C bal DOM leach';
      units       := 'gC m-2 day-1';
      symbol       := 'LDOM';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 84;
With proc[CurrentProc] do
   begin
      name       := 'C bal DOMR leach';
      units       := 'gC m-2 day-1';
      symbol       := 'LDOMR';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 85;
With proc[CurrentProc] do
   begin
      name       := 'N bal DON leach';
      units       := 'gN m-2 day-1';
      symbol       := 'LDON';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 86;
With proc[CurrentProc] do
   begin
      name       := 'N bal DONR leach';
      units       := 'gN m-2 day-1';
      symbol       := 'LDONR';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 87;
With proc[CurrentProc] do
   begin
      name       := 'N bal total N leach';
      units       := 'gN m-2 day-1';
      symbol       := 'LNtot';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 88;
With proc[CurrentProc] do
   begin
      name       := 'P bal PO4 leach';
      units       := 'gP m-2 day-1';
      symbol       := 'LPO4';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 89;
With proc[CurrentProc] do
   begin
      name       := 'thetaN';
      units       := 'gC/gN';
      symbol       := 'thetaN';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 90;
With proc[CurrentProc] do
   begin
      name       := 'thetaP';
      units       := 'gC/gP';
      symbol       := 'thetaP';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 91;
With proc[CurrentProc] do
   begin
      name       := 'N bal Micr NH4 uptake';
      units       := 'g N m-2 day-1';
      symbol       := 'UNH4m';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 92;
With proc[CurrentProc] do
   begin
      name       := 'N bal Micr NO3 uptake';
      units       := 'g N m-2 day-1';
      symbol       := 'UNO3m';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 93;
With proc[CurrentProc] do
   begin
      name       := 'C bal Micr DOM uptake';
      units       := 'g N m-2 day-1';
      symbol       := 'UDOMm';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 94;
With proc[CurrentProc] do
   begin
      name       := 'N bal Micr DON uptake';
      units       := 'g N m-2 day-1';
      symbol       := 'UDONm';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 95;
With proc[CurrentProc] do
   begin
      name       := 'N bal tot Micr N uptake';
      units       := 'g N m-2 day-1';
      symbol       := 'UNmtot';
      parameters       := 13;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='microb C:N';  units:='g C/g N';  symbol:='phiN';
 end;
with par[npar + 2] do
 begin
    name:='NH4 up par m';  units:='g N m-2 day-1';  symbol:='alphaNH4';
 end;
with par[npar + 3] do
 begin
    name:='NH4 1/2sat m';  units:='umol N L-1';  symbol:='kNH4m';
 end;
with par[npar + 4] do
 begin
    name:='NO3 up par m';  units:='g N m-2 day-1';  symbol:='alphaNO3';
 end;
with par[npar + 5] do
 begin
    name:='NO3 1/2sat m';  units:='umol N L-1';  symbol:='kNO3m';
 end;
with par[npar + 6] do
 begin
    name:='DOM up par m';  units:='day-1';  symbol:='alphaDOM';
 end;
with par[npar + 7] do
 begin
    name:='DOM 1/2sat m';  units:='umol N L-1';  symbol:='kDOMm';
 end;
with par[npar + 8] do
 begin
    name:='DOM prod rate';  units:='m2 g-1 N';  symbol:='rrDOM';
 end;
with par[npar + 9] do
 begin
    name:='DOMR prod rate';  units:='m2 g-1 N';  symbol:='rrDOMR';
 end;
with par[npar + 10] do
 begin
    name:='C:N DOM';  units:='gC g-1 N';  symbol:='qDOM';
 end;
with par[npar + 11] do
 begin
    name:='C:N DOMR';  units:='gC g-1 N';  symbol:='qDOMR';
 end;
with par[npar + 12] do
 begin
    name:='Nitrifica rate';  units:='g N m-2';  symbol:='rrNitr';
 end;
with par[npar + 13] do
 begin
    name:='Nitrifica 1/2sat const';  units:='umol N L-1';  symbol:='kNH4Nitr';
 end;
 
CurrentProc := ModelDef.numstate + 96;
With proc[CurrentProc] do
   begin
      name       := 'P bal Micr P uptake';
      units       := 'g P m-2 day-1';
      symbol       := 'UPO4m';
      parameters       := 3;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='microb C:P';  units:='g C/g N';  symbol:='phiP';
 end;
with par[npar + 2] do
 begin
    name:='PO4 up par m';  units:='g P m-2 day-1';  symbol:='alphaPO4';
 end;
with par[npar + 3] do
 begin
    name:='PO4 1/2sat m';  units:='umol P L-1';  symbol:='kPO4m';
 end;
 
CurrentProc := ModelDef.numstate + 97;
With proc[CurrentProc] do
   begin
      name       := 'total C consump';
      units       := 'g C m-2 day-1';
      symbol       := 'MC';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 98;
With proc[CurrentProc] do
   begin
      name       := 'total N consump';
      units       := 'g N m-2 day-1';
      symbol       := 'MN';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 99;
With proc[CurrentProc] do
   begin
      name       := 'total P consump';
      units       := 'g P m-2 day-1';
      symbol       := 'MP';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 100;
With proc[CurrentProc] do
   begin
      name       := 'C efficiency';
      units       := 'none';
      symbol       := 'lambdaC';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 101;
With proc[CurrentProc] do
   begin
      name       := 'N efficiency';
      units       := 'none';
      symbol       := 'lambdaN';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 102;
With proc[CurrentProc] do
   begin
      name       := 'P efficiency';
      units       := 'none';
      symbol       := 'lambdaP';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 103;
With proc[CurrentProc] do
   begin
      name       := 'C bal Microbial resp';
      units       := 'g C m-2 day-1';
      symbol       := 'Rm';
      parameters       := 8;
      ptype       := ptGroup4;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='max C eff';  units:='none';  symbol:='epsilonC';
 end;
with par[npar + 2] do
 begin
    name:='N pref coef';  units:='none';  symbol:='nuN';
 end;
with par[npar + 3] do
 begin
    name:='P pref coef';  units:='none';  symbol:='nuP';
 end;
with par[npar + 4] do
 begin
    name:='metabolic const';  units:='day-1';  symbol:='psi';
 end;
with par[npar + 5] do
 begin
    name:='Q10 microb resp';  units:='none';  symbol:='Q10';
 end;
with par[npar + 6] do
 begin
    name:='opt moisture';  units:='pore fraction';  symbol:='H2Oopt';
 end;
with par[npar + 7] do
 begin
    name:='J paramter';  units:='none';  symbol:='Jmoist';
 end;
with par[npar + 8] do
 begin
    name:='minimum moist';  units:='pore fraction';  symbol:='H2Omin';
 end;
 
CurrentProc := ModelDef.numstate + 104;
With proc[CurrentProc] do
   begin
      name       := 'N bal gross N min';
      units       := 'g N m-2 day-1';
      symbol       := 'RNm';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 105;
With proc[CurrentProc] do
   begin
      name       := 'P bal gross P min';
      units       := 'g P m-2 day-1';
      symbol       := 'RPm';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 106;
With proc[CurrentProc] do
   begin
      name       := 'C bal Phase II tran';
      units       := 'g C m-2 day-1';
      symbol       := 'TIIC';
      parameters       := 1;
      ptype       := ptGroup4;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='transition rate modifier';  units:='none';  symbol:='xiII';
 end;
 
CurrentProc := ModelDef.numstate + 107;
With proc[CurrentProc] do
   begin
      name       := 'N bal Phase II tran';
      units       := 'g C m-2 day-1';
      symbol       := 'TIIN';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 108;
With proc[CurrentProc] do
   begin
      name       := 'P bal Phase II tran';
      units       := 'g C m-2 day-1';
      symbol       := 'TIIP';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 109;
With proc[CurrentProc] do
   begin
      name       := 'C bal Phase II min';
      units       := 'g C m-2 day-1';
      symbol       := 'MIIC';
      parameters       := 1;
      ptype       := ptGroup4;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='mineralize rate modifier';  units:='none';  symbol:='rhoII';
 end;
 
CurrentProc := ModelDef.numstate + 110;
With proc[CurrentProc] do
   begin
      name       := 'N bal Phase II min';
      units       := 'g C m-2 day-1';
      symbol       := 'MIIN';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 111;
With proc[CurrentProc] do
   begin
      name       := 'P bal Phase II min';
      units       := 'g C m-2 day-1';
      symbol       := 'MIIP';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 112;
With proc[CurrentProc] do
   begin
      name       := 'N Bal Non Sym N fix';
      units       := 'g N m-2 day-1';
      symbol       := 'NNSfix';
      parameters       := 3;
      ptype       := ptGroup4;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='lichen rate const';  units:='g N m-2 day-1';  symbol:='betaNfix';
 end;
with par[npar + 2] do
 begin
    name:='soil Nfix rate const';  units:='g2 N g-2 C day-1';  symbol:='gammaNfix';
 end;
with par[npar + 3] do
 begin
    name:='Nfic crit C:N';  units:='g C g-1 N';  symbol:='qSfix';
 end;
 
CurrentProc := ModelDef.numstate + 113;
With proc[CurrentProc] do
   begin
      name       := 'C bal DOM production';
      units       := 'gC m-2 day-1';
      symbol       := 'PDOM';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 114;
With proc[CurrentProc] do
   begin
      name       := 'C bal DOMR production';
      units       := 'gC m-2 day-1';
      symbol       := 'PDOMR';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 115;
With proc[CurrentProc] do
   begin
      name       := 'N bal DON production';
      units       := 'gN m-2 day-1';
      symbol       := 'PDON';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 116;
With proc[CurrentProc] do
   begin
      name       := 'N bal DONR production';
      units       := 'gN m-2 day-1';
      symbol       := 'PDONR';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 117;
With proc[CurrentProc] do
   begin
      name       := 'N bal NItrification';
      units       := 'g N m-2 day-1';
      symbol       := 'Nitr';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 118;
With proc[CurrentProc] do
   begin
      name       := 'delta effort C';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVC';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 119;
With proc[CurrentProc] do
   begin
      name       := 'delta effort NH4';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVNH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 120;
With proc[CurrentProc] do
   begin
      name       := 'delta effort NO3';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVNO3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 121;
With proc[CurrentProc] do
   begin
      name       := 'delta effort DON';
      units       := 'effort g-2 DW day-1';
      symbol       := 'dVDON';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 122;
With proc[CurrentProc] do
   begin
      name       := 'delta effort Nfix';
      units       := 'effort g-2 DW day-1';
      symbol       := 'dVNfix';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 123;
With proc[CurrentProc] do
   begin
      name       := 'delta effort PO4';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVP';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 124;
With proc[CurrentProc] do
   begin
      name       := 'delta effort light';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVI';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 125;
With proc[CurrentProc] do
   begin
      name       := 'delta effort H2O';
      units       := 'effort g-1 DW day-1';
      symbol       := 'dVW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 126;
With proc[CurrentProc] do
   begin
      name       := 'delta effort total';
      units       := 'effort g-2 DW day-1';
      symbol       := 'dVtot';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 127;
With proc[CurrentProc] do
   begin
      name       := 'Total H2O req';
      units       := 'mm day-1';
      symbol       := 'RW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 128;
With proc[CurrentProc] do
   begin
      name       := 'N bal total Ndep';
      units       := 'g N m-2 day-1';
      symbol       := 'Ndeptot';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 129;
With proc[CurrentProc] do
   begin
      name       := 'N bal net N min';
      units       := 'g N m-2 day-1';
      symbol       := 'netNmin';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 130;
With proc[CurrentProc] do
   begin
      name       := 'P bal net P min';
      units       := 'g N m-2 day-1';
      symbol       := 'netPmin';
      parameters       := 0;
      ptype       := ptGroup4;
   end;
 
CurrentProc := ModelDef.numstate + 131;
With proc[CurrentProc] do
   begin
      name       := 'C bal NEP';
      units       := 'g C m-2 day-1';
      symbol       := 'NEP';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 132;
With proc[CurrentProc] do
   begin
      name       := 'N bal Net ecos';
      units       := 'g N m-2 day-1';
      symbol       := 'NENB';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 133;
With proc[CurrentProc] do
   begin
      name       := 'P bal Net ecos';
      units       := 'g P m-2 day-1';
      symbol       := 'NEPB';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 134;
With proc[CurrentProc] do
   begin
      name       := 'C bal budget';
      units       := 'g C m-2 day-1';
      symbol       := 'Cbud';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 135;
With proc[CurrentProc] do
   begin
      name       := 'biomass w full canopy';
      units       := 'g C m-2';
      symbol       := 'BTstar';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 136;
With proc[CurrentProc] do
   begin
      name       := 'Canopy litter';
      units       := 'gr m-2 day-1';
      symbol       := 'LitCan';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 137;
With proc[CurrentProc] do
   begin
      name       := 'optimum canopy cond';
      units       := 'dum';
      symbol       := 'gWstar';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 138;
With proc[CurrentProc] do
   begin
      name       := 'potential water uptake';
      units       := 'mm day-1';
      symbol       := 'UWS';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 139;
With proc[CurrentProc] do
   begin
      name       := 'Williams GPP';
      units       := 'gC m-2 day-1';
      symbol       := 'WmGPP';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 140;
With proc[CurrentProc] do
   begin
      name       := 'Snow melt';
      units       := 'mm day-1';
      symbol       := 'SM';
      parameters       := 5;
      ptype       := ptGroup5;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='short wave coef';  units:='none';  symbol:='SWC';
 end;
with par[npar + 2] do
 begin
    name:='longwave coef';  units:='none';  symbol:='SB';
 end;
with par[npar + 3] do
 begin
    name:='net rad coef';  units:='mm m2 MJ-1';  symbol:='NRC';
 end;
with par[npar + 4] do
 begin
    name:='convective coef';  units:='mm oC-1 day-1';  symbol:='CC';
 end;
with par[npar + 5] do
 begin
    name:='snow cirt temp';  units:='oC';  symbol:='Tcrit';
 end;
 
CurrentProc := ModelDef.numstate + 141;
With proc[CurrentProc] do
   begin
      name       := 'Julian Day';
      units       := 'day';
      symbol       := 'Jday';
      parameters       := 7;
      ptype       := ptGroup5;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='time offset';  units:='days';  symbol:='Tly';
 end;
with par[npar + 2] do
 begin
    name:='start Deg day sum';  units:='Julian Day';  symbol:='JDstart';
 end;
with par[npar + 3] do
 begin
    name:='Deg day bud open';  units:='Deg day';  symbol:='Ddbud';
 end;
with par[npar + 4] do
 begin
    name:='Deg day full canopy';  units:='Deg day';  symbol:='Ddfull';
 end;
with par[npar + 5] do
 begin
    name:='day fall starts';  units:='Julian day';  symbol:='JDfallstart';
 end;
with par[npar + 6] do
 begin
    name:='day fall ends';  units:='Julian day';  symbol:='JDfallend';
 end;
with par[npar + 7] do
 begin
    name:='latitude';  units:='degrees';  symbol:='lat';
 end;
 
CurrentProc := ModelDef.numstate + 142;
With proc[CurrentProc] do
   begin
      name       := 'Day length';
      units       := 'hr day-1';
      symbol       := 'DL';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 143;
With proc[CurrentProc] do
   begin
      name       := 'declination';
      units       := 'radians';
      symbol       := 'delta';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 144;
With proc[CurrentProc] do
   begin
      name       := 'Interception';
      units       := 'mm day-1';
      symbol       := 'Intrcpt';
      parameters       := 0;
      ptype       := ptGroup5;
   end;
 
CurrentProc := ModelDef.numstate + 145;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio CA';
      units       := 'none';
      symbol       := 'LAMBDA_C';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 146;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio light';
      units       := 'none';
      symbol       := 'LAMBDA_I';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 147;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio NH4';
      units       := 'none';
      symbol       := 'LAMBDA_NH4';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 148;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio NO3';
      units       := 'none';
      symbol       := 'LAMBDA_NO3';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 149;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio DON';
      units       := 'none';
      symbol       := 'LAMBDA_DON';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 150;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio Nfix';
      units       := 'none';
      symbol       := 'LAMBDA_Nfix';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 151;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio PO4';
      units       := 'none';
      symbol       := 'LAMBDA_P';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 152;
With proc[CurrentProc] do
   begin
      name       := 'requirement ratio H2O';
      units       := 'none';
      symbol       := 'LAMBDA_W';
      parameters       := 7;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='gain C';  units:='none';  symbol:='Cgain';
 end;
with par[npar + 2] do
 begin
    name:='gain N';  units:='none';  symbol:='Ngain';
 end;
with par[npar + 3] do
 begin
    name:='gain P';  units:='none';  symbol:='Pgain';
 end;
with par[npar + 4] do
 begin
    name:='Crit P effort';  units:='effort g-1 DW';  symbol:='VPcrit';
 end;
with par[npar + 5] do
 begin
    name:='Weathering gain';  units:='g DW effort-1 day-1';  symbol:='gammaP';
 end;
with par[npar + 6] do
 begin
    name:='C cost of P weathering';  units:='g C m-2';  symbol:='gammaPC';
 end;
with par[npar + 7] do
 begin
    name:='N cost of P weathering';  units:='g N m-2';  symbol:='gammaPN';
 end;
 
CurrentProc := ModelDef.numstate + 153;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield C';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUcdVC';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 154;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield W';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUcdVW';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 155;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield I';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUcIdVI';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 156;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield NH4';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUNdVNH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 157;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield NO3';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUNdVNO3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 158;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield DON';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUNdVDON';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 159;
With proc[CurrentProc] do
   begin
      name       := 'marginal yield Nfix';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUNdVNfix';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 160;
With proc[CurrentProc] do
   begin
      name       := 'max marginal yield C';
      units       := 'gC m-2 day-1 effort-1';
      symbol       := 'dUcdVmax';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 161;
With proc[CurrentProc] do
   begin
      name       := 'near root depletion fac';
      units       := 'none';
      symbol       := 'BetaNRD';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 162;
With proc[CurrentProc] do
   begin
      name       := 'between root half dista';
      units       := 'none';
      symbol       := 'rD';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 163;
With proc[CurrentProc] do
   begin
      name       := 'Root Length';
      units       := 'm m-2 soil';
      symbol       := 'RL';
      parameters       := 10;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='specific root length';  units:='m g-1 DW';  symbol:='a_srl';
 end;
with par[npar + 2] do
 begin
    name:='root radius';  units:='mm';  symbol:='rr';
 end;
with par[npar + 3] do
 begin
    name:='diffusion NH4';  units:='m gN L m-3 umol-1 d-1';  symbol:='DNH4';
 end;
with par[npar + 4] do
 begin
    name:='diffusion NO3';  units:='m gN L m-3 umol-1 d-1';  symbol:='DNO3';
 end;
with par[npar + 5] do
 begin
    name:='diffusion DON';  units:='m gN L m-3 umol-1 d-1';  symbol:='DDOM';
 end;
with par[npar + 6] do
 begin
    name:='diffusion PO4';  units:='m gP L m-3 umol-1 d-1';  symbol:='DPO4';
 end;
with par[npar + 7] do
 begin
    name:='Nfix inhibition';  units:='none';  symbol:='NfixI';
 end;
with par[npar + 8] do
 begin
    name:='scaler UCI';  units:='none';  symbol:='sUCI';
 end;
with par[npar + 9] do
 begin
    name:='scaler UCc';  units:='none';  symbol:='sUCc';
 end;
with par[npar + 10] do
 begin
    name:='scaler ET';  units:='none';  symbol:='scET';
 end;

{ Set the total number of drivers in the model. The maximum number of drivers is
  maxdrive, in unit stypes. }
ModelDef.numdrive := 12;

{ Set the names, units, and symbols of the drivers. The maximum length for the
  name, units and symbol is 20 characters. }
 
with drive[1] do
 begin
    name:='total short wave';  units:='MJ m-2 day-1';  symbol:='ISW';
 end;
 
with drive[2] do
 begin
    name:='Temperature min';  units:='oC';  symbol:='Tmin';
 end;
 
with drive[3] do
 begin
    name:='Temperature max';  units:='oC';  symbol:='Tmax';
 end;
 
with drive[4] do
 begin
    name:='carbon dioxide';  units:='ppm';  symbol:='CA';
 end;
 
with drive[5] do
 begin
    name:='NH4 input';  units:='g N m-2';  symbol:='INH4';
 end;
 
with drive[6] do
 begin
    name:='NO3 input';  units:='g N m-2';  symbol:='INO3';
 end;
 
with drive[7] do
 begin
    name:='PO4 input';  units:='g P m-2';  symbol:='IPO4';
 end;
 
with drive[8] do
 begin
    name:='ppt';  units:='mm';  symbol:='rain';
 end;
 
with drive[9] do
 begin
    name:='DOM input';  units:='g N m-2';  symbol:='IDOM';
 end;
 
with drive[10] do
 begin
    name:='DOMR input';  units:='g N m-2';  symbol:='IDOMR';
 end;
 
with drive[11] do
 begin
    name:='apatite input';  units:='g N m-2';  symbol:='IPa';
 end;
 
with drive[12] do
 begin
    name:='Dept of thaw';  units:='m';  symbol:='DOT';
 end;

{ The first numstate processes are the derivatives of the state variables. The
  code sets the names, units and symbols accordingly.}
for i:= 1 to ModelDef.numstate do proc[i].name:='d'+stat[i].name+'dt';
for i:= 1 to ModelDef.numstate do proc[i].units := stat[i].units + 't-1';
for i:= 1 to ModelDef.numstate do proc[i].symbol := 'd' + stat[i].symbol + 'dt';

{ Code to sum up the total number of parameters in the model. Do not change the
  next few lines. }
ModelDef.numparam := 0;
for i := 1 to ModelDef.NumProcess do
  ModelDef.numparam := ModelDef.numparam + proc[i].parameters;

end; // counts procedure


{ A procedure to calculate the value of all states and processes at the current
  time. This function accesses time, state variables and process variables by
  reference, ie it uses the same array as the calling routine. It does not use
  the global variables time, stat and proc because values calculated during
  integration might later be discarded. It does access the global variables par,
  drive and ModelDef directly because those values are not modified.

  The model equations are written using variable names which correspond to the
  actual name instead of using the global arrays (i.e. SoilWater instead of
  stat[7].value). This makes it necessary to switch all values into local
  variables, do all the calculations and then put everything back into the
  global variables. Lengthy but worth it in terms of readability of the code. }

// Choose either GlobalPs, ArcticPs, or none here so the appropriate Ps model is compiled below.
{$DEFINE none}

PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
{$IFDEF GlobalPs}
const
// Global Ps parameters
 x1 = 11.04;             x2 = 0.03;
 x5 = 0.216;             x6 = 0.6;
 x7 = 3.332;             x8 = 0.004;
 x9 = 1.549;             x10 = 1.156;
 gammastar = 0;          kCO2 = 995.4;  }
{$ENDIF}

// Modify constant above (line above "procedure processes..." line )to specify
// which Ps model and it's constants should be compiled. Choosing a Ps model
// automatically includes the Et and Misc constants (i.e. Gem is assumed).

{$IFDEF ArcticPs}
const
// Arctic Ps parameters
x1 = 0.192;	x2 = 0.125;
x5 = 2.196;	x6 = 50.41;
x7 = 0.161;	x8 = 14.78;
x9 = 1.146;
gammastar = 0.468;	kCO2 = 500.3;
{$ENDIF}

{$IFDEF ArcticPs OR GlobalPs}
//const
// General Et parameters
aE1 = 0.0004;    aE2 = 150;  aE3 = 1.21;   aE4 = 6.11262E5;

// Other constants
cp = 1.012E-9; //specific heat air MJ kg-1 oC-1
sigmaSB = 4.9e-9; //stefan-Boltzmann MJ m-2 day-1 K-4
S0 = 117.5; //solar constant MJ m-2 day-1
bHI1 =0.23;
bHI2 =0.48;
mw = 2.99; //kg h2o MJ-1
alphaMS = 2; //mm oC-1 day-1                                 }
{$ENDIF}

var
{ List the variable names you are going to use here. Generally, this list
  includes all the symbols you defined in the procedure counts above. The order
  in which you list them does not matter. }
{States}
BC, dBCdt, 
BN, dBNdt, 
BP, dBPdt, 
VC, dVCdt, 
VNH4, dVNH4dt, 
VNO3, dVNO3dt, 
VP, dVPdt, 
VI, dVIdt, 
VW, dVWdt, 
VDON, dVDONdt, 
VNfix, dVNfixdt, 
DC, dDCdt, 
DN, dDNdt, 
DP, dDPdt, 
WC, dWCdt, 
WN, dWNdt, 
WP, dWPdt, 
SC, dSCdt, 
SN, dSNdt, 
SP, dSPdt, 
ENH4, dENH4dt, 
ENO3, dENO3dt, 
EPO4, dEPO4dt, 
PA, dPAdt, 
Pno, dPnodt, 
Poccl, dPoccldt, 
EDOM, dEDOMdt, 
EDOMR, dEDOMRdt, 
W, dWdt, 
Snow, dSnowdt, 
cumint, dcumintdt, 
RA_C, dRA_Cdt, 
RA_N, dRA_Ndt, 
RA_NH4, dRA_NH4dt, 
RA_NO3, dRA_NO3dt, 
RA_DON, dRA_DONdt, 
RA_Nfix, dRA_Nfixdt, 
RA_P, dRA_Pdt, 
RA_W, dRA_Wdt, 
UA_C, dUA_Cdt, 
UA_NH4, dUA_NH4dt, 
UA_NO3, dUA_NO3dt, 
UA_P, dUA_Pdt, 
UA_I, dUA_Idt, 
UA_W, dUA_Wdt, 
UA_DON, dUA_DONdt, 
UA_Nfix, dUA_Nfixdt, 
cumGPP, dcumGPPdt, 
cumNPP, dcumNPPdt, 
cumlitC, dcumlitCdt, 
cumRm, dcumRmdt, 
cumNup, dcumNupdt, 
cumlitN, dcumlitNdt, 
cumNmin, dcumNmindt, 
cumPup, dcumPupdt, 
cumlitP, dcumlitPdt, 
cumPmin, dcumPmindt, 
cumPpt, dcumPptdt, 
cumET, dcumETdt, 
cumRO, dcumROdt, 
cumNin, dcumNindt, 
cumNout, dcumNoutdt, 
cumPin, dcumPindt, 
cumPout, dcumPoutdt, 
cumNfix, dcumNfixdt, 
DDay, dDDaydt, 
cumNSNfix, dcumNSNfixdt, 

{processes and associated parameters}
Ta, 
BT, 
BA, BAmax, gammaB, a_sla, qC, 
BL, 
BR, 
BW, 
L, 
L_max, 
fc, 
DELTA_E, 
AW, theta_f, theta_w, psi_f, psi_w, z, pore, rho_S, drain, NLsfc, NLe, MBW, IntV, SNH4, etaNH4, SDOM, etaDOM, SDOMR, etaDOMR, etaPO4, ROfNH4, ROfNO3, ROfDOM, ROfDOMR, ROfPO4, 
PaW, 
PO4P, rPaW, Pnomax, rPO4P, rPnoW, rPnoS, rPocclW, 
PnoW, 
PocclW, 
PnoS, 
psiS, 
c_cmax, 
c_cs, 
c_cI, 
c_c, 
UCc, 
UCI, 
UCW, 
UC, gC, kC, TxC, ToC, aC, gI, E0, kI, TxL, ToL, aL, 
UW, gW, kE, c_smax, beta, 
UWp, 
NH4aq, 
UNH4, 
NO3aq, 
UNO3, 
DOM, 
DOMR, 
UDOM, 
UDON, 
UNfix, 
UNtot, gNH4, kNH4, Q10NH4, gNO3, kNO3, Q10NO3, gDOM, kDOM, Q10DOM, gNfix, Q10Nfix, NH4Ccost, NO3Ccost, DONCcost, NfixCcost, 
PO4aq, 
UPO4, gPO4, kPO4, Q10PO4, 
aqN, 
qN, qLN, qWN, qRN, qLNl, qWNl, qRNl, kq, 
aqP, 
qP, qLP, qWP, qRP, qLPl, qWPl, qRPl, 
LitC, mAL, mW, mAR, fracmin, 
LitN, 
LitP, 
LCWC, mCW, mCWex, qWNwl, qWPwl, 
LCWN, 
LCWP, 
LCWCa, omega, 
LCWNa, 
LCWPa, 
RCm, 
RCPt, rmA, rmW, krmW, Q10R, rgrow, 
NUE, 
PUE, 
WUE, 
Vstar, a, tau, Vsc, epsilon, lambda, 
RCg, 
RNg, 
RPg, 
yNH4, 
yNO3, 
yDON, 
yNfix, 
sumUy, 
RC, 
RP, 
RNT, 
RNH4, 
RNO3, 
RDON, 
RNfix, 
PHI, 
Ci, 
VTot, 
VR, 
VL, 
NPP, 
RO, 
LNH4, 
LNO3, 
LDOM, 
LDOMR, 
LDON, 
LDONR, 
LNtot, 
LPO4, 
thetaN, 
thetaP, 
UNH4m, 
UNO3m, 
UDOMm, 
UDONm, 
UNmtot, phiN, alphaNH4, kNH4m, alphaNO3, kNO3m, alphaDOM, kDOMm, rrDOM, rrDOMR, qDOM, qDOMR, rrNitr, kNH4Nitr, 
UPO4m, phiP, alphaPO4, kPO4m, 
MC, 
MN, 
MP, 
lambdaC, 
lambdaN, 
lambdaP, 
Rm, epsilonC, nuN, nuP, psi, Q10, H2Oopt, Jmoist, H2Omin, 
RNm, 
RPm, 
TIIC, xiII, 
TIIN, 
TIIP, 
MIIC, rhoII, 
MIIN, 
MIIP, 
NNSfix, betaNfix, gammaNfix, qSfix, 
PDOM, 
PDOMR, 
PDON, 
PDONR, 
Nitr, 
dVC, 
dVNH4, 
dVNO3, 
dVDON, 
dVNfix, 
dVP, 
dVI, 
dVW, 
dVtot, 
RW, 
Ndeptot, 
netNmin, 
netPmin, 
NEP, 
NENB, 
NEPB, 
Cbud, 
BTstar, 
LitCan, 
gWstar, 
UWS, 
WmGPP, 
SM, SWC, SB, NRC, CC, Tcrit, 
Jday, Tly, JDstart, Ddbud, Ddfull, JDfallstart, JDfallend, lat, 
DL, 
delta, 
Intrcpt, 
LAMBDA_C, 
LAMBDA_I, 
LAMBDA_NH4, 
LAMBDA_NO3, 
LAMBDA_DON, 
LAMBDA_Nfix, 
LAMBDA_P, 
LAMBDA_W, Cgain, Ngain, Pgain, VPcrit, gammaP, gammaPC, gammaPN, 
dUcdVC, 
dUcdVW, 
dUcIdVI, 
dUNdVNH4, 
dUNdVNO3, 
dUNdVDON, 
dUNdVNfix, 
dUcdVmax, 
BetaNRD, 
rD, 
RL, a_srl, rr, DNH4, DNO3, DDOM, DPO4, NfixI, sUCI, sUCc, scET, 

{drivers}
ISW, 
Tmin, 
Tmax, 
CA, 
INH4, 
INO3, 
IPO4, 
rain, 
IDOM, 
IDOMR, 
IPa, 
DOT, 

{Other double}
theta, 
dum, 
PImax, 
PCmax, 
gCstar, 
resp, 
divsr, 
psi2, 
Pc2, 
PL2, 
phi2, 
ActL, 
Ymax, 
R, 
S, 
delVC, 
delVW, 
delVI, 
delVNH4, 
delVNO3, 
delVDON, 
delVNfix, 
VCdum, 
VNH4dum, 
VNO3dum, 
VPdum, 
VIdum, 
VWdum, 
VDONdum, 
VNfixdum

:double; {Final double}
{Other integers}
npar, j, jj, kk, ll, tnum:integer;


{ Functions or procedures }
Procedure Assess_Allometry(VC,VNH4,VNO3,VP,VI,VW,VDON,VNfix:double);
begin {allometry}
   {Biomass (DW)}
   BT:= BC/qC;
   {canopy allocation}
   VL:=VC+VI;
   {Root allocation}
   VR:=VW+VNH4+VNO3+VDON+VP+VNfix;
   {find BTStar = biomass if canopy were full}
   dum:=BAmax-gammaB*BT-(1-fc)*BAmax*gammaB*VL;
   BTstar:=-dum+sqrt(sqr(dum)+4*gammaB*BAmax*BT);
   BTstar:=BTstar/(2*gammaB);
   {individual tissues}
   BA:=BAmax*gammaB*BTstar/(BAmax+gammaB*BTstar);
   BW:=BTstar-BA;
   {leaf biomass}
   BL:=fc*VL*BA;
   {leaf area}
   L:=a_sla*BL;
   {full-canopy leaf area}
   L_max:=a_sla*VL*BA;
   {root biomass}
   BR:=VR*BA;
   {root length}
   RL:=BR*a_srl;
   {average between-root half distance}
   rD:=sqrt(z/pi/RL);
   {near-root depletion factor}
   BetaNRD:=sqr(rD)*(0.5*ln(rD)-0.25);
   BetaNRD:=BetaNRD - sqr(rr)*(0.5*ln(rr)-0.25);
   BetaNRD:=BetaNRD*2/(sqr(rD)-sqr(rr));
   BetaNRD:=ln(rr)-BetaNRD;
   BetaNRD:=BetaNRD/(2*pi);
end;  {allometry}

Function light_capture(VI:double):double;
var UCI:double;
begin {light capture}
   PImax:=gI*(VI-Vsc)/VL;
   if DL>0 then UCI:=sUCI*DL*PImax*ln((PImax+E0*Isw/DL)/(PImax+E0*Isw*exp(-kI*L)/DL))/kI
   else UCI:=0;
   UCI:=UCI*1.6*(Ca-60)/(Ca+120);
   light_capture:=UCI*exp(aL*(Ta-ToL))*power((TxL-Ta)/(TxL-ToL),aL*(TxL-ToL));
end;{light capture}

Function CO2_H2O_uptake(VC,VW:double):double;
var theta:double;
begin
     {7.775 = mm H2O/m/MPa =rhoa*Cp/(lambda*gamma); c_cmax in m/hr}
     {0.000335=1/1.6*(12e-6 gC/umolCO2)*(1000L/m3)*(1/22.4 mol/L)}
   c_cmax:=c_smax*L; {max canopy conductance}
   UWp:=scET*7.775*DL*c_cmax*DELTA_e; {Max ET}
   UWs:=ActL*gW*RL*((VW-Vsc)/VR)*(PsiS-Psi_w)*(1-exp(-kE*UWp)); if UWs<0 then UWs:=0;
   if DL>0 then c_cs:=UWs/( scET*7.775*DL*DELTA_e)
   else c_cs:=0; {soil-water-limited Canopy conductance}
   if c_cs>c_cmax then c_cs:=c_cmax;
   PCmax:=gC*((VC-Vsc)/VL)*L;
   PCmax:=PCmax*exp(aC*(Ta-ToC))*power((TxC-Ta)/(TxC-ToC),aC*(TxC-ToC));
   if c_cs > 0 then 
      begin
         theta:=Ca-kC-PCmax/(0.000335*c_cs);
         Ci:=0.5*(theta+sqrt(sqr(theta)+4*kC*Ca));
      end
   else Ci:=Ca;
   CO2_H2O_uptake:=sUCc*0.000335*DL*c_cs*(Ca-Ci); //0.95 scaling factor
end;{CO2_H2O_uptake}

function uptake(Cbar,Vi,km,D,g,Q10,psi:double):double;
var
  V,Cs:double;
begin
   V:=g*((Vi-Vsc)/VR)*RL*power(Q10,Ta/10);
   Cs:=km-Cbar+V*betaNRD/(D*RL*psi);
   Cs:=0.5*(-Cs+sqrt(sqr(Cs)+4*km*Cbar));
   Uptake:=ActL*V*Cs/(km+Cs);
end;{uptake}

function Nfix(Vi,g,Q10:double):double;
begin
Nfix:=ActL*g*((Vi-Vsc)/VR)*power(Q10,Ta/10)*RL;
end;{Nfix}

Procedure assess_dV(V1,V2,V3,V4,V5,V6,V7,V8:double; var VD1,VD2,VD3,VD4,VD5,VD6,VD7,VD8,delV1:double);
var 
   dum:double;
begin
   delV1:=V1-(Vsc+0.99*(V1-Vsc));
   if delV1<1e-5 then delV1:=-1e-5;
   dum:=delV1/(1-7*Vsc-V1);
   VD1:=V1-delV1;
   VD2:=V2;//+(V2-Vsc)*dum;
   VD3:=V3;//+(V3-Vsc)*dum;
   VD4:=V4;//+(V4-Vsc)*dum;
   VD5:=V5;//+(V5-Vsc)*dum;
   VD6:=V6;//+(V6-Vsc)*dum;
   VD7:=V7;//+(V7-Vsc)*dum;
   VD8:=V8;//+(V8-Vsc)*dum;
end; {assess_dV}

function water_tension(W:double):double;
var
   b,psi_i,m,n,theta,theta_i:double;
begin
{Clapp & Hornberger 1978 WRR 14:601 water tension, assume air entry at 0.92 WFPS}
theta:= W/1000/z;
theta_i:=0.92*pore;
b:=-ln(psi_f/psi_w)/ln(theta_f/theta_w);
Psi_i:=psi_f*power(theta_i/theta_f,-b); 
m:=(11.5-b)*Psi_i/0.0736; 
n:= 0.84-0.08*b/(11.5-b); 
if theta<theta_i then water_tension:=psi_f*power(theta/theta_f,-b)
else water_tension:= -m*(theta/pore-n)*(theta/pore-1);
end; {water_tension}

Function ACM(L:double):double;
var PsiD,gC,theta,beta,Ci,PD,PII:double;
begin
   psiD:=PsiS-Psi_W;
   if psiD<1e-6 then psiD:=1e-6;
   gC:=power(psiD,0.0006575)/(0.2653+(Tmax-Tmin)/2);
   beta:=2.155*BL*qLN*exp(0.06013*Tmax)/gC;
   theta:=CA+216.9196-beta;
   Ci:=theta+sqrt(sqr(theta)-4*(CA*216.9196-beta*217.9));
   Ci:=Ci/2;
   PD:=gC*(CA-Ci);
   E0:=4.309*sqr(L)/(1.062+sqr(L));
   PII:=E0*ISW*PD/(E0*ISW+PD);
   ACM:=PII*(0.01416*DL+0.1548);
end; {ACM}


begin
{ Copy the drivers from the global array, drive, into the local variables. }
ISW := tdrive[1].value;
Tmin := tdrive[2].value;
Tmax := tdrive[3].value;
CA := tdrive[4].value;
INH4 := tdrive[5].value;
INO3 := tdrive[6].value;
IPO4 := tdrive[7].value;
rain := tdrive[8].value;
IDOM := tdrive[9].value;
IDOMR := tdrive[10].value;
IPa := tdrive[11].value;
DOT := tdrive[12].value;

{ Copy the state variables from the global array into the local variables. }
BC := tstat[1].value;
BN := tstat[2].value;
BP := tstat[3].value;
VC := tstat[4].value;
VNH4 := tstat[5].value;
VNO3 := tstat[6].value;
VP := tstat[7].value;
VI := tstat[8].value;
VW := tstat[9].value;
VDON := tstat[10].value;
VNfix := tstat[11].value;
DC := tstat[12].value;
DN := tstat[13].value;
DP := tstat[14].value;
WC := tstat[15].value;
WN := tstat[16].value;
WP := tstat[17].value;
SC := tstat[18].value;
SN := tstat[19].value;
SP := tstat[20].value;
ENH4 := tstat[21].value;
ENO3 := tstat[22].value;
EPO4 := tstat[23].value;
PA := tstat[24].value;
Pno := tstat[25].value;
Poccl := tstat[26].value;
EDOM := tstat[27].value;
EDOMR := tstat[28].value;
W := tstat[29].value;
Snow := tstat[30].value;
cumint := tstat[31].value;
RA_C := tstat[32].value;
RA_N := tstat[33].value;
RA_NH4 := tstat[34].value;
RA_NO3 := tstat[35].value;
RA_DON := tstat[36].value;
RA_Nfix := tstat[37].value;
RA_P := tstat[38].value;
RA_W := tstat[39].value;
UA_C := tstat[40].value;
UA_NH4 := tstat[41].value;
UA_NO3 := tstat[42].value;
UA_P := tstat[43].value;
UA_I := tstat[44].value;
UA_W := tstat[45].value;
UA_DON := tstat[46].value;
UA_Nfix := tstat[47].value;
cumGPP := tstat[48].value;
cumNPP := tstat[49].value;
cumlitC := tstat[50].value;
cumRm := tstat[51].value;
cumNup := tstat[52].value;
cumlitN := tstat[53].value;
cumNmin := tstat[54].value;
cumPup := tstat[55].value;
cumlitP := tstat[56].value;
cumPmin := tstat[57].value;
cumPpt := tstat[58].value;
cumET := tstat[59].value;
cumRO := tstat[60].value;
cumNin := tstat[61].value;
cumNout := tstat[62].value;
cumPin := tstat[63].value;
cumPout := tstat[64].value;
cumNfix := tstat[65].value;
DDay := tstat[66].value;
cumNSNfix := tstat[67].value;

{ And now copy the parameters into the local variables. No need to copy the
  processes from the global array into local variables. Process values will be
  calculated by this procedure.

  Copy the parameters for each process separately using the function ParCount
  to keep track of the number of parameters in the preceeding processes.
  npar now contains the number of parameters in the preceding processes.
  copy the value of the first parameter of this process into it's local
  variable }
npar:=ParCount(ModelDef.numstate + 3);
BAmax := par[npar + 1].value;
gammaB := par[npar + 2].value;
a_sla := par[npar + 3].value;
qC := par[npar + 4].value;

npar:=ParCount(ModelDef.numstate + 11);
theta_f := par[npar + 1].value;
theta_w := par[npar + 2].value;
psi_f := par[npar + 3].value;
psi_w := par[npar + 4].value;
z := par[npar + 5].value;
pore := par[npar + 6].value;
rho_S := par[npar + 7].value;
drain := par[npar + 8].value;
NLsfc := par[npar + 9].value;
NLe := par[npar + 10].value;
MBW := par[npar + 11].value;
IntV := par[npar + 12].value;
SNH4 := par[npar + 13].value;
etaNH4 := par[npar + 14].value;
SDOM := par[npar + 15].value;
etaDOM := par[npar + 16].value;
SDOMR := par[npar + 17].value;
etaDOMR := par[npar + 18].value;
etaPO4 := par[npar + 19].value;
ROfNH4 := par[npar + 20].value;
ROfNO3 := par[npar + 21].value;
ROfDOM := par[npar + 22].value;
ROfDOMR := par[npar + 23].value;
ROfPO4 := par[npar + 24].value;
 
npar:=ParCount(ModelDef.numstate + 13);
rPaW := par[npar + 1].value;
Pnomax := par[npar + 2].value;
rPO4P := par[npar + 3].value;
rPnoW := par[npar + 4].value;
rPnoS := par[npar + 5].value;
rPocclW := par[npar + 6].value;
 
npar:=ParCount(ModelDef.numstate + 25);
gC := par[npar + 1].value;
kC := par[npar + 2].value;
TxC := par[npar + 3].value;
ToC := par[npar + 4].value;
aC := par[npar + 5].value;
gI := par[npar + 6].value;
E0 := par[npar + 7].value;
kI := par[npar + 8].value;
TxL := par[npar + 9].value;
ToL := par[npar + 10].value;
aL := par[npar + 11].value;
 
npar:=ParCount(ModelDef.numstate + 26);
gW := par[npar + 1].value;
kE := par[npar + 2].value;
c_smax := par[npar + 3].value;
beta := par[npar + 4].value;
 
npar:=ParCount(ModelDef.numstate + 37);
gNH4 := par[npar + 1].value;
kNH4 := par[npar + 2].value;
Q10NH4 := par[npar + 3].value;
gNO3 := par[npar + 4].value;
kNO3 := par[npar + 5].value;
Q10NO3 := par[npar + 6].value;
gDOM := par[npar + 7].value;
kDOM := par[npar + 8].value;
Q10DOM := par[npar + 9].value;
gNfix := par[npar + 10].value;
Q10Nfix := par[npar + 11].value;
NH4Ccost := par[npar + 12].value;
NO3Ccost := par[npar + 13].value;
DONCcost := par[npar + 14].value;
NfixCcost := par[npar + 15].value;
 
npar:=ParCount(ModelDef.numstate + 39);
gPO4 := par[npar + 1].value;
kPO4 := par[npar + 2].value;
Q10PO4 := par[npar + 3].value;
 
npar:=ParCount(ModelDef.numstate + 41);
qLN := par[npar + 1].value;
qWN := par[npar + 2].value;
qRN := par[npar + 3].value;
qLNl := par[npar + 4].value;
qWNl := par[npar + 5].value;
qRNl := par[npar + 6].value;
kq := par[npar + 7].value;
 
npar:=ParCount(ModelDef.numstate + 43);
qLP := par[npar + 1].value;
qWP := par[npar + 2].value;
qRP := par[npar + 3].value;
qLPl := par[npar + 4].value;
qWPl := par[npar + 5].value;
qRPl := par[npar + 6].value;
 
npar:=ParCount(ModelDef.numstate + 44);
mAL := par[npar + 1].value;
mW := par[npar + 2].value;
mAR := par[npar + 3].value;
fracmin := par[npar + 4].value;
 
npar:=ParCount(ModelDef.numstate + 47);
mCW := par[npar + 1].value;
mCWex := par[npar + 2].value;
qWNwl := par[npar + 3].value;
qWPwl := par[npar + 4].value;
 
npar:=ParCount(ModelDef.numstate + 50);
omega := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 54);
rmA := par[npar + 1].value;
rmW := par[npar + 2].value;
krmW := par[npar + 3].value;
Q10R := par[npar + 4].value;
rgrow := par[npar + 5].value;
 
npar:=ParCount(ModelDef.numstate + 58);
a := par[npar + 1].value;
tau := par[npar + 2].value;
Vsc := par[npar + 3].value;
epsilon := par[npar + 4].value;
lambda := par[npar + 5].value;
 
npar:=ParCount(ModelDef.numstate + 95);
phiN := par[npar + 1].value;
alphaNH4 := par[npar + 2].value;
kNH4m := par[npar + 3].value;
alphaNO3 := par[npar + 4].value;
kNO3m := par[npar + 5].value;
alphaDOM := par[npar + 6].value;
kDOMm := par[npar + 7].value;
rrDOM := par[npar + 8].value;
rrDOMR := par[npar + 9].value;
qDOM := par[npar + 10].value;
qDOMR := par[npar + 11].value;
rrNitr := par[npar + 12].value;
kNH4Nitr := par[npar + 13].value;
 
npar:=ParCount(ModelDef.numstate + 96);
phiP := par[npar + 1].value;
alphaPO4 := par[npar + 2].value;
kPO4m := par[npar + 3].value;
 
npar:=ParCount(ModelDef.numstate + 103);
epsilonC := par[npar + 1].value;
nuN := par[npar + 2].value;
nuP := par[npar + 3].value;
psi := par[npar + 4].value;
Q10 := par[npar + 5].value;
H2Oopt := par[npar + 6].value;
Jmoist := par[npar + 7].value;
H2Omin := par[npar + 8].value;
 
npar:=ParCount(ModelDef.numstate + 106);
xiII := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 109);
rhoII := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 112);
betaNfix := par[npar + 1].value;
gammaNfix := par[npar + 2].value;
qSfix := par[npar + 3].value;
 
npar:=ParCount(ModelDef.numstate + 140);
SWC := par[npar + 1].value;
SB := par[npar + 2].value;
NRC := par[npar + 3].value;
CC := par[npar + 4].value;
Tcrit := par[npar + 5].value;
 
npar:=ParCount(ModelDef.numstate + 141);
Tly := par[npar + 1].value;
JDstart := par[npar + 2].value;
Ddbud := par[npar + 3].value;
Ddfull := par[npar + 4].value;
JDfallstart := par[npar + 5].value;
JDfallend := par[npar + 6].value;
lat := par[npar + 7].value;
 
npar:=ParCount(ModelDef.numstate + 152);
Cgain := par[npar + 1].value;
Ngain := par[npar + 2].value;
Pgain := par[npar + 3].value;
VPcrit := par[npar + 4].value;
gammaP := par[npar + 5].value;
gammaPC := par[npar + 6].value;
gammaPN := par[npar + 7].value;
 
npar:=ParCount(ModelDef.numstate + 163);
a_srl := par[npar + 1].value;
rr := par[npar + 2].value;
DNH4 := par[npar + 3].value;
DNO3 := par[npar + 4].value;
DDOM := par[npar + 5].value;
DPO4 := par[npar + 6].value;
NfixI := par[npar + 7].value;
sUCI := par[npar + 8].value;
sUCc := par[npar + 9].value;
scET := par[npar + 10].value;
 
 
{ Enter the equations to calculate the processes here, using the local variable
  names defined above. }

Ndeptot:=INH4+INO3+IDOM/qdom+IDOMR/qDOMR;

{correct round off for V}
Vstar:=0;
if VC>Vsc then begin VC:=VC-Vsc; Vstar:=Vstar+VC; end else VC:=0;
if VNH4>Vsc then begin VNH4:=VNH4-Vsc; Vstar:=Vstar+VNH4; end else VNH4:=0;
if VNO3>Vsc then begin VNO3:=VNO3-Vsc; Vstar:=Vstar+VNO3; end else VNO3:=0;
if VP>Vsc then begin VP:=VP-Vsc; Vstar:=Vstar+VP; end else VP:=0;
if VW>Vsc then begin VW:=VW-Vsc; Vstar:=Vstar+VW; end else VW:=0; 
if VI>Vsc then begin VI:=VI-Vsc; Vstar:=Vstar+VI; end else VI:=0;
if VNfix>Vsc then begin VNfix:=VNfix-Vsc; Vstar:=Vstar+VNfix; end else VNfix:=0;
if VDON>Vsc then begin VDON:=VDON-Vsc; Vstar:=Vstar+VDON; end else VDON:=0;

vstar:=vstar/(1-8*Vsc);

VC:=Vsc+VC/Vstar;            tstat[4].value:=VC;
VNH4:=Vsc+VNH4/Vstar;        tstat[5].value:=VNH4;
VNO3:=Vsc+VNO3/Vstar;        tstat[6].value:=VNO3;
VP:=Vsc+VP/Vstar;            tstat[7].value:=VP;
VI:=Vsc+VI/Vstar;            tstat[8].value:=VI;
VW:=Vsc+VW/Vstar;            tstat[9].value:=VW;
VDON:=Vsc+VDON/Vstar;        tstat[10].value:=VDON;
VNfix:=Vsc+VNfix/Vstar;      tstat[11].value:=VNfix;
Vtot:=VC+VNH4+VNO3+VP+VW+VI+VNfix+VDON;

{active layer fraction}
ActL:=DOT/z; if ActL>1 then ActL:=1;

{Julian Day}
Jday:= tan(pi*((time+Tly)/365.25 -0.50001));
Jday:= arctan(Jday)+pi/2;
Jday:=trunc((365.25/pi)*Jday+0.9);
{Day length}
delta:=0.1303*pi*cos(2*pi*(Jday-172)/365);
DL:=-tan(lat*pi/180)*tan(delta);
if DL>1 then DL:=1;
if DL<-1 then DL:=-1;
DL:= 24*arccos(DL)/pi;
{Average temperature}
Ta:=(Tmax+Tmin)/2;
{soil water tension}
PsiS:=water_tension(W);
{Peak vapor pressure deficit}
DELTA_e:=6.1078e-4*(exp(17.269*Tmax/(237.3+Tmax))-exp(17.269*Tmin/(237.3+Tmin)));
{available water}
AW:=(W/(1000*z)-theta_w)/(theta_f-theta_w); if AW>1 then AW:=1; if AW<0 then AW:=0;
{aqueous NH4,  14e-6 converts umoles to g N}
NH4aq:= ENH4 - 14E-6*W*etaNH4 - z*rho_s*SNH4; 
NH4aq:= NH4aq + sqrt(sqr(NH4aq)+ 56e-6*W*ENH4*etaNH4);
NH4aq:=NH4aq/(28e-6*W);
{aqueous NO3}
NO3aq:=ENO3/(14e-6*W);
{DOM}
DOM:= EDOM - 12E-6*W*etaDOM - z*rho_s*SDOM;
DOM:= DOM + sqrt(sqr(DOM)+ 48e-6*W*EDOM*etaDOM);
DOM:=DOM/(24e-6*W);
{DOMR}
DOMR:= EDOMR - 12E-6*W*etaDOMR - z*rho_s*SDOMR;
DOMR:= DOMR + sqrt(sqr(DOMR)+ 48e-6*W*EDOMR*etaDOMR);
DOMR:=DOMR/(24e-6*W);
{aqueous PO4}
PO4aq:= EPO4/(31e-6*(1000*etaPO4*z+W)); 

{Canopy fraction; included in allometric table}
if Jday <JDstart then Dday:=0;
S:=(Ddbud+Ddfull)/2;
dum:=(JDfallend+JDfallstart)/2;
fc:=fracmin;
if (Dday-Ddbud)*(S-Dday)>=0 then
          fc:=fracmin + (1-fracmin)*2*sqr((Dday-Ddbud)/(Ddfull-Ddbud))
else
  if (Ddfull-Dday)*(Dday-S)>=0 then
          fc:=1 - (1-fracmin)*2*sqr((Ddfull-Dday)/(Ddfull-Ddbud))
  else
     if (Dday-Ddfull)*(JDfallstart-Jday)>=0 then fc:=1
     else
        if (Jday-JDfallstart)*(dum-Jday)>=0 then
            fc:=1-(1-fracmin)*2*sqr((JDay-JDfallstart)/(JDfallend-JDfallstart))
        else
          if (JDfallend-Jday)*(Jday-dum)>=0 then
             fc:=fracmin+(1-fracmin)*2*sqr((JDfallend-JDay)/(JDfallend-JDfallstart));

{Assess dUCdVC}
assess_dV(VC,VNH4,VNO3,VP,VI,VW,VDON,VNfix, VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum,delVC);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUcdVC:=CO2_H2O_uptake(VCdum,VWdum); // final calculation done below

{Assess dUCdVW}
assess_dV(VW,VNH4,VNO3,VP,VI,VC,VDON,VNfix, VWdum,VNH4dum,VNO3dum,VPdum,VIdum,VCdum,VDONdum,VNfixdum,delVW);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUcdVW:=CO2_H2O_uptake(VCdum,VWdum); // final calculation done below

{Assess dUCIdVI}
assess_dV(VI,VNH4,VNO3,VP,VC,VW,VDON,VNfix, VIdum,VNH4dum,VNO3dum,VPdum,VCdum,VWdum,VDONdum,VNfixdum,delVI);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUcIdVI:=light_capture(VIdum);// final calculation done below

{Assess dUNdVNH4}
assess_dV(VNH4,VC,VNO3,VP,VI,VW,VDON,VNfix, VNH4dum,VCdum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum,delVNH4);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUNdVNH4:=uptake(NH4aq,VNH4dum,kNH4,DNH4,gNH4,Q10NH4,14e-3);// final calc done below

{Assess dUNdVNO3}
assess_dV(VNO3,VNH4,VC,VP,VI,VW,VDON,VNfix, VNO3dum,VNH4dum,VCdum,VPdum,VIdum,VWdum,VDONdum,VNfixdum,delVNO3);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUNdVNO3:=uptake(NO3aq,VNO3dum,kNO3,DNO3,gNO3,Q10NO3,14e-3);// final calculation done below

{Assess dUNdVDON}
assess_dV(VDON,VNH4,VNO3,VP,VI,VW,VC,VNfix, VDONdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VCdum,VNfixdum,delVDON);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUNdVDON:=uptake(DOM,VDONdum,kDOM,DDOM,gDOM,Q10DOM,12e-3);
dUNdVDON:=dUNdVDON/qDOM;// final calculation done below

{Assess dUNdVNfix}
assess_dV(VNfix,VNH4,VNO3,VP,VI,VW,VDON,VC, VNfixdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VCdum,delVNfix);
Assess_Allometry(VCdum,VNH4dum,VNO3dum,VPdum,VIdum,VWdum,VDONdum,VNfixdum);
dUNdVNfix:=Nfix(VNfixdum,gNfix,Q10Nfix); // final calculation done below

Assess_Allometry(VC,VNH4,VNO3,VP,VI,VW,VDON,VNfix);

UCI:= light_capture(VI);            //light-limited photosynthesis
UC:=CO2_H2O_uptake(VC,VW);          //CO2&H2O-limited photosynthesis

UNH4:=uptake(NH4aq,VNH4,kNH4,DNH4,gNH4,Q10NH4,14e-3);
UNO3:=uptake(NO3aq,VNO3,kNO3,DNO3,gNO3,Q10NO3,14e-3);
UDOM:=uptake(DOM,VDON,kDOM,DDOM,gDOM,Q10DOM,12e-3);
UDON:=UDOM/qDOM;
UNfix:=Nfix(VNfix,gNfix,Q10Nfix);
UPO4:=uptake(PO4aq,VP,kPO4,DPO4,gPO4,Q10PO4,31e-3);

dUcdVC:=(UC-dUcdVC)/delVC;
dUcdVW:=(UC-dUcdVW)/delVW;
dUCIdVI:=(UCI-dUCIdVI)/delVI;
if UCI<UC then dUcdVmax:=dUCIdVI else dUCdVmax:=max(dUCdVC,dUCdVW); //1/marg. cost C
if dUcdVmax<1e-6 then dUcdVmax:= 1e-6;

dUNdVNH4:=(UNH4-dUNdVNH4)/delVNH4;
dUNdVNO3:=(UNO3-dUNdVNO3)/delVNO3;
dUNdVDON:=(UDON-dUNdVDON)/delVDON;
dUNdVNfix:=(UNfix-dUNdVNfix)/delVNfix;

UCW:= UC+(dUCdVC-dUcdVW)*beta;     //H2O-limited photosynthesis   
UCC:= UC+(dUcdVW-dUCdVC)*beta;     //C-limited photosynthesis

UC:= min(UC,UCI);                 //Photosynthesis

if DL>0 then Ci:=kC*UCI/(sUCC*DL*PCmax-UCI)
else S:=2*Ca;
if (S>Ca) or (S<0) then c_cI:=c_cmax
else c_cI:= UCI/(0.000335*sUCC*DL*(Ca-S)); {light-limited Canopy conductance}

c_c:=min(c_cI,c_cs);         {actual Canopy conductance}
UW:= scET*7.775*DL*c_c*DELTA_e;         //H2O uptake = ET


{N yield}
yNH4:=dUNdVNH4/(1+dUNdVNH4*NH4Ccost/dUCdVmax); {yield}
yNO3:=dUNdVNO3/(1+dUNdVNO3*NO3Ccost/dUCdVmax); {yield}
yDON:=dUNdVDON/(1+dUNdVDON*DONCcost/dUCdVmax);{yield}
yNfix:=dUNdVNfix/(1+dUNdVNfix*NfixCcost/(dUCdVmax*exp(NfixI*power(BW,1/3)))); {yield}

{total N uptake}
UNtot:=UNH4+UNO3+UDON+UNfix;

{Williams GPP, ACM}
WmGPP:=ACM(L);

{WUE}
if UW>0 then WUE:=UC/UW else WUE:=0;


PaW:= rPAW*PA;                            {apatite weathering}
if VP>VPcrit then 
   PaW := PaW + gammaP*(VP-VPcrit)*PA*BA;{plant mediated apatite weathering}
PaW:=ActL*PaW;
PO4P:=ActL*rPO4P*PO4aq*(z*rho_s*Pnomax-Pno);   {PO4 precipitation to nonoccluded}
PnoW:=ActL*rPnoW*Pno;                          {release of non occl to PO4}
PocclW:=ActL*rPocclW*Poccl;                    {weathering on occluded to non occl}
PnoS:=ActL*rPnoS*Pno;                          {stabilization of non occ to occl}

{optimum element concentrations; included in allometric table}
qN:=(qLN*BL+qWN*BW+qRN*BR)/BT;
qP:=(qLP*BL+qWP*BW+qRP*BR)/BT;
{actual element concentrations; included in allometric table}
aqN:=BN/BT;
aqP:=BP/BT;

{litter}
LitCan:=0;
dum:=(JDfallend+JDfallstart)/2;
if (Jday-JDfallstart)*(dum-Jday)>=0 then
    LitCan:=4*(1-fracmin)*VL*BA*(Jday-JDfallstart)/sqr(JDfallend-JDfallstart);
if (JDfallend-Jday)*(Jday-dum)>=0 then
    LitCan:=4*(1-fracmin)*VL*BA*(JDfallend-Jday)/sqr(JDfallend-JDfallstart);
LitCan:=LitCan+mAL*VL*BA*fracmin;

LCWC:=mCW*power(BW,mCWex);
LCWN:=BN*LCWC*qWNwl/(qN*BT);
LCWP:=BP*LCWC*qWPwl/(qP*BT);
LCWC:=BC*LCWC/BT;

LitC:=BC*(mW*BA*BW/(BA+BW) + maR*BR + LitCan)/BT; 
if VP > VPcrit then LitC := LitC + gammaPC*gammaP*(VP-VPcrit)*BA;
LitN:=BN*(mW*BA*BW/(BA+BW)*qWNl + maR*BR*qRNl + litCan*qLNl)/(qN*BT); 
if VP > VPcrit then LitC := LitC + gammaPN*gammaP*(VP-VPcrit)*BA;
LitP:=BP*(mW*BA*BW/(BA+BW)*qWPl + maR*BR*qRPl + litCan*qLPl)/(qP*BT);

{NUE & PUE}
NUE:=LitC/LitN;
PUE:=LitC/LitP;

{Resource requirement}
{for maintenance}
RCm:=power(Q10R,Ta/10)*(rmA*(qLN*BL+qRN*BR)+rmW*qWN*BW*exp(-BW*krmW));

NPP:=UC-RCm-UNH4*NH4Ccost-UNO3*NO3Ccost-UDON*DONCcost-UNfix*NfixCcost;
if NPP>0 then NPP:=NPP/(1+rgrow);
RCPt:=RCm+UNH4*NH4Ccost+UNO3*NO3Ccost+UDON*DONCcost+UNfix*NfixCcost+rgrow*NPP;

{Resource requirement, replacement growth}
RCg:= LitC*(1+rgrow)*power(sqrt((BN/(BT*qN))*(BP/(BT*qP))),kq);
RNg:= LitN*power(qN*BT/BN,kq);
RPg:= LitP*power(qP*BT/BP,kq);

{total requirement}
{nitrogen}
RNT:= RNg;
{Phosphorus}
RP:= RPg;
{carbon}
RC:=RCm + RCg + UNH4*NH4Ccost+UNO3*NO3Ccost+UDON*DONCcost+UNfix*NfixCcost;

RC:=Cgain*RC;
RNT:= Ngain*RNT;
RP:=Pgain*RP;

{ET requirement; based on Ci =beta CA }

  
RW:= scET*7.775*DL*c_cs*DELTA_e;
if RW<0 then RW:=0;

ymax:=max(max(max(YNH4,YNO3),YDON),YNfix);

if ymax>0 then
   begin
      RNH4 :=exp(-sqr((YNH4-ymax)/(Ymax*epsilon)));
      RNO3 :=exp(-sqr((YNO3-ymax)/(Ymax*epsilon)));
      RDON :=exp(-sqr((YDON-ymax)/(Ymax*epsilon)));
      RNfix:=exp(-sqr((YNfix-ymax)/(Ymax*epsilon)));
   end
else
   begin 
      RNH4 :=RA_NH4;
      RNO3 :=RA_NO3;
      RDON :=RA_DON;
      RNfix:=RA_Nfix;
   end;

sumUy:=RNH4+RNO3+RDON+RNfix;
RNH4:=RNT*RNH4/sumUy;
RNO3:=RNT*RNO3/sumUy;
RDON:=RNT*RDON/sumUy;
RNfix:=RNT*RNfix/sumUy;

R:=1;
S:= 1;
phi:= 100;
while abs(S-phi)>1e-3 do
begin
S:=R;
LAMBDA_C:=(lambda*RA_C +UA_C + 0.0001*qC)/(RA_C +lambda*UA_C + 0.0001*qC*s);
LAMBDA_NH4:=(lambda*RA_NH4 +UA_NH4 + 0.0001*qN)/(RA_NH4 +lambda*UA_NH4 + 0.0001*qN*s);
LAMBDA_NO3:=(lambda*RA_NO3 +UA_NO3 + 0.0001*qN)/(RA_NO3 +lambda*UA_NO3 + 0.0001*qN*s);
LAMBDA_DON:=(lambda*RA_DON +UA_DON + 0.0001*qN)/(RA_DON +lambda*UA_DON + 0.0001*qN*s);
LAMBDA_Nfix:=(lambda*RA_Nfix +UA_Nfix + 0.0001*qN)/(RA_Nfix +lambda*UA_Nfix + 0.0001*qN*s);
LAMBDA_P:=(lambda*RA_P +UA_P + 0.0001*qP)/(RA_P +lambda*UA_P + 0.0001*qP*s);
LAMBDA_I:=(lambda*RA_C +UA_I + 0.0001*qC)/(RA_C +lambda*UA_I + 0.0001*qC*s);
LAMBDA_W:=(lambda*RA_C +UA_W + 0.0001*qC)/(RA_C +lambda*UA_W + 0.0001*qC*s);
phi:=1;
phi:=phi*power(1/LAMBDA_C,VC);
phi:=phi*power(1/LAMBDA_NH4,VNH4);
phi:=phi*power(1/LAMBDA_NO3,VNO3);
phi:=phi*power(1/LAMBDA_DON,VDON);
phi:=phi*power(1/LAMBDA_Nfix,VNfix);
phi:=phi*power(1/LAMBDA_P,VP);
phi:=phi*power(1/LAMBDA_I,VI);
phi:=phi*power(1/LAMBDA_W,VW);

R:=phi;
end;

if LAMBDA_C>0    then dVC  :=a*(ln(PHI*LAMBDA_C)*VC)       
else begin dVC:=0;    LAMBDA_C:=1/phi; end;
if LAMBDA_NH4>0  then dVNH4  :=a*(ln(PHI*LAMBDA_NH4)*VNH4)   
else begin dVNH4:=0;  LAMBDA_NH4:=1/phi; end;
if LAMBDA_NO3>0  then dVNO3  :=a*(ln(PHI*LAMBDA_NO3)*VNO3)   
else begin dVNO3:=0;  LAMBDA_NO3:=1/phi; end;
if LAMBDA_DON>0  then dVDON  :=a*(ln(PHI*LAMBDA_DON)*VDON)   
else begin dVDON:=0;  LAMBDA_DON:=1/phi; end;
if LAMBDA_Nfix>0 then dVNfix :=a*(ln(PHI*LAMBDA_Nfix)*VNfix) 
else begin dVNfix:=0; LAMBDA_Nfix:=1/phi; end;
if LAMBDA_P>0    then dVP  :=a*(ln(PHI*LAMBDA_P)*VP)       
else begin dVP:=0;    LAMBDA_P:=1/phi; end;
if LAMBDA_I>0    then dVI:=a*(ln(PHI*LAMBDA_I)*VI)       
else begin dVI:=0;    LAMBDA_I:=1/phi; end;
if LAMBDA_W>0    then dVW  :=a*(ln(PHI*LAMBDA_W)*VW)       
else begin dVW:=0;    LAMBDA_W:=1/phi; end;

dVtot:=dVC+dVNH4+dVNO3+dVDON+dVNfix+dVP+dVI+dVW;


{leaching}
RO:=ActL*DRAIN*(W - theta_f*1000*z); {mm H2O = kg H2O/m2 = L/m2}
if RO<0 then RO:=0;
LNH4:=RO*ROfNH4*NH4aq*14e-6;
LNO3:=RO*ROfNO3*NO3aq*14e-6;
LDOM:=RO*ROfDOM*DOM*12e-6;
LDOMR:=RO*ROfDOMR*DOMR*12e-6;
LDON:=LDOM/qDOM;
LDONR:=LDOMR/qDOMR;
LPO4:=RO*ROfPO4*PO4aq*31e-6;
LNtot:=LNH4+LNO3+LDON+LDONR;

{microbial processes}
R:=W/(z*1000*pore);
S:=H2Oopt-H2Omin;
resp:=1-sqr((H2Oopt-R)/S)/(1+Jmoist*((R-H2Omin)/S));
resp:=ActL*resp*power(Q10,Ta/10);
if resp<1e-9 then resp:=1e-9;
LCWCa:=omega*resp*WC;
LCWNa:=omega*resp*WN;
LCWPa:=omega*resp*WP;
TIIC:=resp*xiII*DC;
TIIN:=TIIC/phiN;
TIIP:=TIIC/phiP;
MIIC:=resp*rhoII*SC;
MIIN:=resp*rhoII*SN;
MIIP:=resp*rhoII*SP;
NNSfix:=resp*betaNfix*power(BW,NLe)*(1-MBW/sqrt(sqr(MBW)+sqr(BW)));
if DC/DN>qSfix then NNSfix:=NNSfix+resp*gammaNfix*(DC/DN-qSfix)*DC;
thetaN:=phiN/epsilonC;
thetaP:=phiP/epsilonC;
UNH4m:=resp*psi*alphaNH4*sqr(DC)*NH4AQ/(thetaN*DN*(kNH4m+NH4AQ));
UNO3m:=resp*psi*alphaNO3*sqr(DC)*NO3AQ/(thetaN*DN*(kNO3m+NO3AQ));
UPO4m:=resp*psi*alphaPO4*sqr(DC)*PO4AQ/(thetaP*DP*(kPO4m+PO4AQ));
UDOMm:=resp*psi*alphaDOM*DOM/(kDOMm+DOM);
UDONm:=UDOMm/qDOM;
UNmtot:=UDONm+UNH4m+UNO3m;
MC:=resp*psi*DC+UDOMm;
MN:=resp*nuN*DN+UNH4m+UNO3m+UDONm;
MP:=resp*nuP*DP+UPO4m;
divsr:=thetaN*MN*thetaP*MP+MC*thetaP*MP+MC*thetaN*MN;
LambdaC:=epsilonC*thetaN*MN*thetaP*MP/divsr;
LambdaN:=MC*thetaP*MP/divsr;
LambdaP:=MC*thetaN*MN/divsr;
Rm:=MC*(1-LambdaC);
RNm:=MN*(1-LambdaN);
RPm:=MP*(1-LambdaP);
PDOM:=resp*rrDOM*DC*DN;
PDOMR:=resp*rrDOMR*DC*DN;
PDON:=PDOM/qDOM;
PDONR:=PDOMR/qDOMR;
Nitr:= resp*rrNitr*NH4aq/(kNH4Nitr+NH4aq);
netNmin:=RNm-UNH4m-UNO3m+PDON-UDONm;
netPmin:=RPm-UPO4m;

NEP:=NPP-Rm;
NENB:=INH4+INO3+UNfix+IDOM/qDOM+IDOMR/qDOMR-LNH4-LNO3-LDOM/qDOM-LDOMR/qDOMR;
NEPB:=IPO4-LPO4;
Cbud:=NPP-Rm+IDOM+IDOMR-LDOM-LDOMR;

Sm:=0;
Intrcpt:=0;

if CalculateDiscrete then
   begin
     {time step size}
      R:=FmOptions.RunOptions.discretestep;
     {Degree day}//Needs to be modified to match Delbart and Picard 2007 model
      if Jday >=JDstart then Dday:=Dday+max(Ta,0)*R
      else Dday:=0;

     {snow melt}
      if Ta > 2.5 then SM:=0 else SM:=Ta-2.5;   {snow sfc temperature}
      SM:=SWC*ISW +4.9e-9*(SB*power(Ta,4)-power(SM,4));
      if Ta>CC then SM:=NRC*SM+CC*Ta else SM:=0;
      if SM>Snow*R
      then SM:=snow*R;
      if SM<0 then SM:=0;
      dum:= min(SM*R,ActL*(1000*pore*z-W));
      W:=W + dum;
      CumRO:=CumRO+SM*R-dum;
      Snow:= snow - SM*R;

      {precipitation}
      cumPpt:=cumPpt+rain*R;
      if rain > IntV*(NLsfc*power(BW,NLe)*(1-MBW/sqrt(sqr(MBW)+sqr(BW)))+L) then
         begin
            rain:= rain - intV*(NLsfc*power(BW,NLe)*(1-MBW/sqrt(sqr(MBW)+sqr(BW)))+L);
            Intrcpt:= intV*(NLsfc*power(BW,NLe)*(1-MBW/sqrt(sqr(MBW)+sqr(BW)))+L);
         end
      else
         begin
            Intrcpt:=rain;
            rain:=0;
         end;
      cumint:=cumint+intrcpt*R;
      cumET:=cumET +intrcpt*R;

      if Ta>Tcrit then 
        begin
          dum:= min(rain*R,ActL*(1000*pore*z-W));
          W:=W + dum;
          CumRO:=CumRO+rain*R-dum;
        end
      else snow:=snow+ rain*R;

   end;    
if CalculateDiscrete then
begin
// Add any discrete processes here
end; //discrete processes


{ Now calculate the derivatives of the state variables. If the holdConstant
  portion of the state variable is set to true then set the derivative equal to
  zero. }
if (tstat[1].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dBCdt := 0
else
 dBCdt:=NPP +UDOM - LitC - LCWC;
 
if (tstat[2].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dBNdt := 0
else
 dBNdt:=UNH4+UNO3+UDOM/qDOM+UNfix-LitN-LCWN;
 
if (tstat[3].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dBPdt := 0
else
 dBPdt:=UPO4-LitP-LCWP;
 
if (tstat[4].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVCdt := 0
else
 dVCdt  :=dVC;
 
if (tstat[5].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVNH4dt := 0
else
 dVNH4dt  :=dVNH4;
 
if (tstat[6].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVNO3dt := 0
else
 dVNO3dt  :=dVNO3;
 
if (tstat[7].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVPdt := 0
else
 dVPdt  :=dVP;
 
if (tstat[8].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVIdt := 0
else
 dVIdt  :=dVI;
 
if (tstat[9].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVWdt := 0
else
 dVWdt  :=dVW;
 
if (tstat[10].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVDONdt := 0
else
 dVDONdt  :=dVDON;
 
if (tstat[11].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVNfixdt := 0
else
 dVNfixdt :=dVNfix;
 
if (tstat[12].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDCdt := 0
else
 dDCdt:=LITC+LCWCa+UDOMm-Rm-PDOM-PDOMR-TIIC;
 
if (tstat[13].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDNdt := 0
else
 dDNdt:=LITN+LCWNa+UNH4m+UNO3m+UDONm-RNm-PDON-PDONR-TIIN+NNSfix;
 
if (tstat[14].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDPdt := 0
else
 dDPdt:=LITP+LCWPa+UPO4m-RPm-TIIP;
 
if (tstat[15].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dWCdt := 0
else
 dWCdt:=LCWC-LCWCa;
 
if (tstat[16].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dWNdt := 0
else
 dWNdt:=LCWN-LCWNa;
 
if (tstat[17].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dWPdt := 0
else
 dWPdt:=LCWP-LCWPa;
 
if (tstat[18].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dSCdt := 0
else
 dSCdt:=TIIC-MIIC;
 
if (tstat[19].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dSNdt := 0
else
 dSNdt:=TIIN-MIIN;
 
if (tstat[20].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dSPdt := 0
else
 dSPdt:=TIIP-MIIP;
 
if (tstat[21].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dENH4dt := 0
else
 dENH4dt:=INH4-UNH4-LNH4-UNH4m+RNm-Nitr+MIIN;
 
if (tstat[22].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dENO3dt := 0
else
 dENO3dt:=INO3-UNO3-LNO3-UNO3m+Nitr;
 
if (tstat[23].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dEPO4dt := 0
else
 dEPO4dt:=PaW+PnoW+IPO4-UPO4-LPO4-UPO4m+RPm-PO4P+MIIP;
 
if (tstat[24].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dPAdt := 0
else
 dPAdt:=IPa-PaW;
 
if (tstat[25].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dPnodt := 0
else
 dPnodt:=PocclW+PO4P-PnoW-PnoS;
 
if (tstat[26].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dPoccldt := 0
else
 dPoccldt:=PnoS-PocclW;
 
if (tstat[27].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dEDOMdt := 0
else
 dEDOMdt:=IDOM-UDOM-LDOM+PDOM-UDOMm;
 
if (tstat[28].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dEDOMRdt := 0
else
 dEDOMRdt:=IDOMR-LDOMR+PDOMR;
 
if (tstat[29].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dWdt := 0
else
 dWdt:=-UW-RO;
 
if (tstat[30].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dSnowdt := 0
else
 dSnowdt:=0 ;
 
if (tstat[31].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumintdt := 0
else
 dcumintdt:=0;
 
if (tstat[32].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_Cdt := 0
else
 dRA_Cdt:=tau*(RC - RA_C);
 
if (tstat[33].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_Ndt := 0
else
 dRA_Ndt:=tau*(RNT - RA_N);
 
if (tstat[34].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_NH4dt := 0
else
 dRA_NH4dt:=tau*(RNH4-RA_NH4);
 
if (tstat[35].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_NO3dt := 0
else
 dRA_NO3dt:=tau*(RNO3-RA_NO3);
 
if (tstat[36].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_DONdt := 0
else
 dRA_DONdt:=tau*(RDON-RA_DON);
 
if (tstat[37].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_Nfixdt := 0
else
 dRA_Nfixdt:=tau*(RNfix-RA_Nfix);
 
if (tstat[38].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_Pdt := 0
else
 dRA_Pdt:=tau*(RP - RA_P);
 
if (tstat[39].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dRA_Wdt := 0
else
 dRA_Wdt:=tau*(RW - RA_W);
 
if (tstat[40].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_Cdt := 0
else
 dUA_Cdt:=tau*(UCc - UA_C);
 
if (tstat[41].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_NH4dt := 0
else
 dUA_NH4dt:=tau*(UNH4 - UA_NH4);
 
if (tstat[42].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_NO3dt := 0
else
 dUA_NO3dt:=tau*(UNO3 - UA_NO3);
 
if (tstat[43].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_Pdt := 0
else
 dUA_Pdt:=tau*(UPO4 - UA_P);
 
if (tstat[44].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_Idt := 0
else
 dUA_Idt:=tau*(UCI - UA_I);
 
if (tstat[45].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_Wdt := 0
else
 dUA_Wdt:=tau*(UCW - UA_W);
 
if (tstat[46].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_DONdt := 0
else
 dUA_DONdt:=tau*(UDON - UA_DON);
 
if (tstat[47].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dUA_Nfixdt := 0
else
 dUA_Nfixdt:=tau*(UNfix - UA_Nfix);
 
if (tstat[48].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumGPPdt := 0
else
 dcumGPPdt:=UC;
 
if (tstat[49].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNPPdt := 0
else
 dcumNPPdt:=NPP;
 
if (tstat[50].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumlitCdt := 0
else
 dcumlitCdt:=litC+LCWC;
 
if (tstat[51].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumRmdt := 0
else
 dcumRmdt:=Rm+MIIC;
 
if (tstat[52].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNupdt := 0
else
 dcumNupdt:=UNH4+UNO3+UDOM/qDOM+UNfix;
 
if (tstat[53].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumlitNdt := 0
else
 dcumlitNdt:=LitN+LCWN;
 
if (tstat[54].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNmindt := 0
else
 dcumNmindt:=-UNH4m-UNO3m-UDONm+RNm+PDON+PDONR+MIIN;
 
if (tstat[55].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumPupdt := 0
else
 dcumPupdt:=UPO4;
 
if (tstat[56].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumlitPdt := 0
else
 dcumlitPdt:=LitP+LCWP;
 
if (tstat[57].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumPmindt := 0
else
 dcumPmindt:=-UPO4m+RPm+MIIP;
 
if (tstat[58].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumPptdt := 0
else
 dcumPptdt:=0;
 
if (tstat[59].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumETdt := 0
else
 dcumETdt:=UW;
 
if (tstat[60].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumROdt := 0
else
 dcumROdt:=RO;
 
if (tstat[61].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNindt := 0
else
 dcumNindt:=INH4+INO3+UNfix+NNSfix+IDOM/qDOM+IDOMR/qDOMR;
 
if (tstat[62].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNoutdt := 0
else
 dcumNoutdt:=LNH4+LNO3+LDOM/qDOM+LDOMR/qDOMR;
 
if (tstat[63].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumPindt := 0
else
 dcumPindt:=IPO4+PAW+PnoW-PO4P;
 
if (tstat[64].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumPoutdt := 0
else
 dcumPoutdt:=LPO4;
 
if (tstat[65].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNfixdt := 0
else
 dcumNfixdt:=UNfix;
 
if (tstat[66].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDDaydt := 0
else
 dDdaydt:=0;
 
if (tstat[67].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dcumNSNfixdt := 0
else
 dcumNSNfixdt:=NNSfix;
 

{ Now that the calculations are complete, assign everything back into the arrays
  so the rest of the code can access the values calculated here. (Local variables
  are destroyed at the end of the procedure).

  Put the state variables back into the global arrays in case the state variable
  was manually changed in this procedure (e.g. discrete state variables or steady state
  calculations).   }
tstat[1].value := BC;
tstat[2].value := BN;
tstat[3].value := BP;
tstat[4].value := VC;
tstat[5].value := VNH4;
tstat[6].value := VNO3;
tstat[7].value := VP;
tstat[8].value := VI;
tstat[9].value := VW;
tstat[10].value := VDON;
tstat[11].value := VNfix;
tstat[12].value := DC;
tstat[13].value := DN;
tstat[14].value := DP;
tstat[15].value := WC;
tstat[16].value := WN;
tstat[17].value := WP;
tstat[18].value := SC;
tstat[19].value := SN;
tstat[20].value := SP;
tstat[21].value := ENH4;
tstat[22].value := ENO3;
tstat[23].value := EPO4;
tstat[24].value := PA;
tstat[25].value := Pno;
tstat[26].value := Poccl;
tstat[27].value := EDOM;
tstat[28].value := EDOMR;
tstat[29].value := W;
tstat[30].value := Snow;
tstat[31].value := cumint;
tstat[32].value := RA_C;
tstat[33].value := RA_N;
tstat[34].value := RA_NH4;
tstat[35].value := RA_NO3;
tstat[36].value := RA_DON;
tstat[37].value := RA_Nfix;
tstat[38].value := RA_P;
tstat[39].value := RA_W;
tstat[40].value := UA_C;
tstat[41].value := UA_NH4;
tstat[42].value := UA_NO3;
tstat[43].value := UA_P;
tstat[44].value := UA_I;
tstat[45].value := UA_W;
tstat[46].value := UA_DON;
tstat[47].value := UA_Nfix;
tstat[48].value := cumGPP;
tstat[49].value := cumNPP;
tstat[50].value := cumlitC;
tstat[51].value := cumRm;
tstat[52].value := cumNup;
tstat[53].value := cumlitN;
tstat[54].value := cumNmin;
tstat[55].value := cumPup;
tstat[56].value := cumlitP;
tstat[57].value := cumPmin;
tstat[58].value := cumPpt;
tstat[59].value := cumET;
tstat[60].value := cumRO;
tstat[61].value := cumNin;
tstat[62].value := cumNout;
tstat[63].value := cumPin;
tstat[64].value := cumPout;
tstat[65].value := cumNfix;
tstat[66].value := DDay;
tstat[67].value := cumNSNfix;

{  Put all process values into process variable array. The first numstate
  processes are the derivatives of the state variables (Calculated above).}
tproc[1].value := dBCdt;
tproc[2].value := dBNdt;
tproc[3].value := dBPdt;
tproc[4].value := dVCdt;
tproc[5].value := dVNH4dt;
tproc[6].value := dVNO3dt;
tproc[7].value := dVPdt;
tproc[8].value := dVIdt;
tproc[9].value := dVWdt;
tproc[10].value := dVDONdt;
tproc[11].value := dVNfixdt;
tproc[12].value := dDCdt;
tproc[13].value := dDNdt;
tproc[14].value := dDPdt;
tproc[15].value := dWCdt;
tproc[16].value := dWNdt;
tproc[17].value := dWPdt;
tproc[18].value := dSCdt;
tproc[19].value := dSNdt;
tproc[20].value := dSPdt;
tproc[21].value := dENH4dt;
tproc[22].value := dENO3dt;
tproc[23].value := dEPO4dt;
tproc[24].value := dPAdt;
tproc[25].value := dPnodt;
tproc[26].value := dPoccldt;
tproc[27].value := dEDOMdt;
tproc[28].value := dEDOMRdt;
tproc[29].value := dWdt;
tproc[30].value := dSnowdt;
tproc[31].value := dcumintdt;
tproc[32].value := dRA_Cdt;
tproc[33].value := dRA_Ndt;
tproc[34].value := dRA_NH4dt;
tproc[35].value := dRA_NO3dt;
tproc[36].value := dRA_DONdt;
tproc[37].value := dRA_Nfixdt;
tproc[38].value := dRA_Pdt;
tproc[39].value := dRA_Wdt;
tproc[40].value := dUA_Cdt;
tproc[41].value := dUA_NH4dt;
tproc[42].value := dUA_NO3dt;
tproc[43].value := dUA_Pdt;
tproc[44].value := dUA_Idt;
tproc[45].value := dUA_Wdt;
tproc[46].value := dUA_DONdt;
tproc[47].value := dUA_Nfixdt;
tproc[48].value := dcumGPPdt;
tproc[49].value := dcumNPPdt;
tproc[50].value := dcumlitCdt;
tproc[51].value := dcumRmdt;
tproc[52].value := dcumNupdt;
tproc[53].value := dcumlitNdt;
tproc[54].value := dcumNmindt;
tproc[55].value := dcumPupdt;
tproc[56].value := dcumlitPdt;
tproc[57].value := dcumPmindt;
tproc[58].value := dcumPptdt;
tproc[59].value := dcumETdt;
tproc[60].value := dcumROdt;
tproc[61].value := dcumNindt;
tproc[62].value := dcumNoutdt;
tproc[63].value := dcumPindt;
tproc[64].value := dcumPoutdt;
tproc[65].value := dcumNfixdt;
tproc[66].value := dDDaydt;
tproc[67].value := dcumNSNfixdt;

{ Now the remaining processes. Be sure to number the processes the same here as
  you did in the procedure counts above. }
tproc[ModelDef.numstate + 1].value := Ta;
tproc[ModelDef.numstate + 2].value := BT;
tproc[ModelDef.numstate + 3].value := BA;
tproc[ModelDef.numstate + 4].value := BL;
tproc[ModelDef.numstate + 5].value := BR;
tproc[ModelDef.numstate + 6].value := BW;
tproc[ModelDef.numstate + 7].value := L;
tproc[ModelDef.numstate + 8].value := L_max;
tproc[ModelDef.numstate + 9].value := fc;
tproc[ModelDef.numstate + 10].value := DELTA_E;
tproc[ModelDef.numstate + 11].value := AW;
tproc[ModelDef.numstate + 12].value := PaW;
tproc[ModelDef.numstate + 13].value := PO4P;
tproc[ModelDef.numstate + 14].value := PnoW;
tproc[ModelDef.numstate + 15].value := PocclW;
tproc[ModelDef.numstate + 16].value := PnoS;
tproc[ModelDef.numstate + 17].value := psiS;
tproc[ModelDef.numstate + 18].value := c_cmax;
tproc[ModelDef.numstate + 19].value := c_cs;
tproc[ModelDef.numstate + 20].value := c_cI;
tproc[ModelDef.numstate + 21].value := c_c;
tproc[ModelDef.numstate + 22].value := UCc;
tproc[ModelDef.numstate + 23].value := UCI;
tproc[ModelDef.numstate + 24].value := UCW;
tproc[ModelDef.numstate + 25].value := UC;
tproc[ModelDef.numstate + 26].value := UW;
tproc[ModelDef.numstate + 27].value := UWp;
tproc[ModelDef.numstate + 28].value := NH4aq;
tproc[ModelDef.numstate + 29].value := UNH4;
tproc[ModelDef.numstate + 30].value := NO3aq;
tproc[ModelDef.numstate + 31].value := UNO3;
tproc[ModelDef.numstate + 32].value := DOM;
tproc[ModelDef.numstate + 33].value := DOMR;
tproc[ModelDef.numstate + 34].value := UDOM;
tproc[ModelDef.numstate + 35].value := UDON;
tproc[ModelDef.numstate + 36].value := UNfix;
tproc[ModelDef.numstate + 37].value := UNtot;
tproc[ModelDef.numstate + 38].value := PO4aq;
tproc[ModelDef.numstate + 39].value := UPO4;
tproc[ModelDef.numstate + 40].value := aqN;
tproc[ModelDef.numstate + 41].value := qN;
tproc[ModelDef.numstate + 42].value := aqP;
tproc[ModelDef.numstate + 43].value := qP;
tproc[ModelDef.numstate + 44].value := LitC;
tproc[ModelDef.numstate + 45].value := LitN;
tproc[ModelDef.numstate + 46].value := LitP;
tproc[ModelDef.numstate + 47].value := LCWC;
tproc[ModelDef.numstate + 48].value := LCWN;
tproc[ModelDef.numstate + 49].value := LCWP;
tproc[ModelDef.numstate + 50].value := LCWCa;
tproc[ModelDef.numstate + 51].value := LCWNa;
tproc[ModelDef.numstate + 52].value := LCWPa;
tproc[ModelDef.numstate + 53].value := RCm;
tproc[ModelDef.numstate + 54].value := RCPt;
tproc[ModelDef.numstate + 55].value := NUE;
tproc[ModelDef.numstate + 56].value := PUE;
tproc[ModelDef.numstate + 57].value := WUE;
tproc[ModelDef.numstate + 58].value := Vstar;
tproc[ModelDef.numstate + 59].value := RCg;
tproc[ModelDef.numstate + 60].value := RNg;
tproc[ModelDef.numstate + 61].value := RPg;
tproc[ModelDef.numstate + 62].value := yNH4;
tproc[ModelDef.numstate + 63].value := yNO3;
tproc[ModelDef.numstate + 64].value := yDON;
tproc[ModelDef.numstate + 65].value := yNfix;
tproc[ModelDef.numstate + 66].value := sumUy;
tproc[ModelDef.numstate + 67].value := RC;
tproc[ModelDef.numstate + 68].value := RP;
tproc[ModelDef.numstate + 69].value := RNT;
tproc[ModelDef.numstate + 70].value := RNH4;
tproc[ModelDef.numstate + 71].value := RNO3;
tproc[ModelDef.numstate + 72].value := RDON;
tproc[ModelDef.numstate + 73].value := RNfix;
tproc[ModelDef.numstate + 74].value := PHI;
tproc[ModelDef.numstate + 75].value := Ci;
tproc[ModelDef.numstate + 76].value := VTot;
tproc[ModelDef.numstate + 77].value := VR;
tproc[ModelDef.numstate + 78].value := VL;
tproc[ModelDef.numstate + 79].value := NPP;
tproc[ModelDef.numstate + 80].value := RO;
tproc[ModelDef.numstate + 81].value := LNH4;
tproc[ModelDef.numstate + 82].value := LNO3;
tproc[ModelDef.numstate + 83].value := LDOM;
tproc[ModelDef.numstate + 84].value := LDOMR;
tproc[ModelDef.numstate + 85].value := LDON;
tproc[ModelDef.numstate + 86].value := LDONR;
tproc[ModelDef.numstate + 87].value := LNtot;
tproc[ModelDef.numstate + 88].value := LPO4;
tproc[ModelDef.numstate + 89].value := thetaN;
tproc[ModelDef.numstate + 90].value := thetaP;
tproc[ModelDef.numstate + 91].value := UNH4m;
tproc[ModelDef.numstate + 92].value := UNO3m;
tproc[ModelDef.numstate + 93].value := UDOMm;
tproc[ModelDef.numstate + 94].value := UDONm;
tproc[ModelDef.numstate + 95].value := UNmtot;
tproc[ModelDef.numstate + 96].value := UPO4m;
tproc[ModelDef.numstate + 97].value := MC;
tproc[ModelDef.numstate + 98].value := MN;
tproc[ModelDef.numstate + 99].value := MP;
tproc[ModelDef.numstate + 100].value := lambdaC;
tproc[ModelDef.numstate + 101].value := lambdaN;
tproc[ModelDef.numstate + 102].value := lambdaP;
tproc[ModelDef.numstate + 103].value := Rm;
tproc[ModelDef.numstate + 104].value := RNm;
tproc[ModelDef.numstate + 105].value := RPm;
tproc[ModelDef.numstate + 106].value := TIIC;
tproc[ModelDef.numstate + 107].value := TIIN;
tproc[ModelDef.numstate + 108].value := TIIP;
tproc[ModelDef.numstate + 109].value := MIIC;
tproc[ModelDef.numstate + 110].value := MIIN;
tproc[ModelDef.numstate + 111].value := MIIP;
tproc[ModelDef.numstate + 112].value := NNSfix;
tproc[ModelDef.numstate + 113].value := PDOM;
tproc[ModelDef.numstate + 114].value := PDOMR;
tproc[ModelDef.numstate + 115].value := PDON;
tproc[ModelDef.numstate + 116].value := PDONR;
tproc[ModelDef.numstate + 117].value := Nitr;
tproc[ModelDef.numstate + 118].value := dVC;
tproc[ModelDef.numstate + 119].value := dVNH4;
tproc[ModelDef.numstate + 120].value := dVNO3;
tproc[ModelDef.numstate + 121].value := dVDON;
tproc[ModelDef.numstate + 122].value := dVNfix;
tproc[ModelDef.numstate + 123].value := dVP;
tproc[ModelDef.numstate + 124].value := dVI;
tproc[ModelDef.numstate + 125].value := dVW;
tproc[ModelDef.numstate + 126].value := dVtot;
tproc[ModelDef.numstate + 127].value := RW;
tproc[ModelDef.numstate + 128].value := Ndeptot;
tproc[ModelDef.numstate + 129].value := netNmin;
tproc[ModelDef.numstate + 130].value := netPmin;
tproc[ModelDef.numstate + 131].value := NEP;
tproc[ModelDef.numstate + 132].value := NENB;
tproc[ModelDef.numstate + 133].value := NEPB;
tproc[ModelDef.numstate + 134].value := Cbud;
tproc[ModelDef.numstate + 135].value := BTstar;
tproc[ModelDef.numstate + 136].value := LitCan;
tproc[ModelDef.numstate + 137].value := gWstar;
tproc[ModelDef.numstate + 138].value := UWS;
tproc[ModelDef.numstate + 139].value := WmGPP;
tproc[ModelDef.numstate + 140].value := SM;
tproc[ModelDef.numstate + 141].value := Jday;
tproc[ModelDef.numstate + 142].value := DL;
tproc[ModelDef.numstate + 143].value := delta;
tproc[ModelDef.numstate + 144].value := Intrcpt;
tproc[ModelDef.numstate + 145].value := LAMBDA_C;
tproc[ModelDef.numstate + 146].value := LAMBDA_I;
tproc[ModelDef.numstate + 147].value := LAMBDA_NH4;
tproc[ModelDef.numstate + 148].value := LAMBDA_NO3;
tproc[ModelDef.numstate + 149].value := LAMBDA_DON;
tproc[ModelDef.numstate + 150].value := LAMBDA_Nfix;
tproc[ModelDef.numstate + 151].value := LAMBDA_P;
tproc[ModelDef.numstate + 152].value := LAMBDA_W;
tproc[ModelDef.numstate + 153].value := dUcdVC;
tproc[ModelDef.numstate + 154].value := dUcdVW;
tproc[ModelDef.numstate + 155].value := dUcIdVI;
tproc[ModelDef.numstate + 156].value := dUNdVNH4;
tproc[ModelDef.numstate + 157].value := dUNdVNO3;
tproc[ModelDef.numstate + 158].value := dUNdVDON;
tproc[ModelDef.numstate + 159].value := dUNdVNfix;
tproc[ModelDef.numstate + 160].value := dUcdVmax;
tproc[ModelDef.numstate + 161].value := BetaNRD;
tproc[ModelDef.numstate + 162].value := rD;
tproc[ModelDef.numstate + 163].value := RL;

end;  // End of processes procedure


       { Do not make any modifications to code below this line. }
{****************************************************************************}


{This function counts the parameters in all processes less than processnum.}
function ParCount(processnum:integer) : integer;
var
 NumberofParams, counter : integer;
begin
  NumberofParams := 0;
  for counter := ModelDef.numstate + 1 to processnum - 1 do
         NumberofParams := NumberofParams + proc[counter].parameters;
  ParCount := NumberofParams;
end; // end of parcount function

{ This procedure supplies the derivatives of the state variables to the
  integrator. Since the integrator deals only with the values of the variables
  and not there names, units or the state field HoldConstant, this procedure
  copies the state values into a temporary state array and copies the value of
  HoldConstant into the temporary state array and passes this temporary state
  array to the procedure processes. }
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
var
   i:integer;
   tempproc:processarray;
   tempstate:statearray;
begin
   tempstate := stat;  // Copy names, units and HoldConstant to tempstate
  // Copy current values of state variables into tempstate
   for i := 1 to ModelDef.numstate do tempstate[i].value := statevalue[i];
  // Calculate the process values
   processes(t, drt, tdrive, tpar, tempstate, tempproc, false);
  // Put process values into dydt array to get passed back to the integrator.
   for i:= 1 to ModelDef.numstate do dydt[i]:=tempproc[i].value;
end;  // end of derivs procedure

end.
